# coding=utf-8

import arrow
from django import template
from django.template import defaultfilters
from django.conf import settings
from django.utils import html

from review.staff import const as staff_const

register = template.Library()


@register.simple_tag(takes_context=True)
def person_review_link(context, person_review, text=None):
    language = context['language']
    href = _person_review_href(person_review)
    if text is None:
        text = href
    elif text == '_DEFAULT_':
        text = {
            'ru': 'открыть карточку сотрудника',
            'en': 'open the person card',
        }[language]
    result = _make_link(
        href=href,
        text=text,
    )
    return result


@register.simple_tag
def calibration_href(calibration):
    return '{url}/calibrations/{calibration_id}'.format(
        url=settings.FRONTEND_URL,
        calibration_id=calibration['id'],
    )


@register.simple_tag
def person_review_href(person_review):
    return _person_review_href(person_review)


def _person_review_href(person_review):
    return '{url}/reviews/{review_id}/{login}/{person_review_id}'.format(
        url=settings.FRONTEND_URL,
        review_id=person_review['review']['id'],
        login=person_review['person']['login'],
        person_review_id=person_review['id'],
    )


@register.simple_tag
def review_link(review, text=None):
    href = '{url}/reviews/{review_id}/'.format(
        url=settings.FRONTEND_URL,
        review_id=review['id'],
    )
    text = text or href
    return _make_link(
        href=href,
        text=text,
    )


@register.simple_tag
def person_avatar_img(person, size=50):
    login = person['login']
    src = '{url}/api/v1/user/{login}/avatar/{size}.jpg'.format(
        url=settings.CENTER_URL,
        login=login,
        size=size,
    )
    result = html.format_html('<img src="{src}" />', src=src)
    return result


@register.simple_tag
def person_avatar_src(person, size=50):
    login = person['login']
    return '{url}/api/v1/user/{login}/avatar/{size}.jpg'.format(
        url=settings.CENTER_URL,
        login=login,
        size=size,
    )


@register.simple_tag
def staff_link(person):
    href = '{url}/{login}'.format(
        url=settings.STAFF_URL,
        login=person['login'],
    )
    text = '{first} {last}'.format(
        first=person['first_name'],
        last=person['last_name'],
    )
    link = _make_link(href, text)
    return html.format_html('{link} ({login})'.format(
        link=link,
        login=person['login'],
    ))


@register.simple_tag
def staff_person(person):
    text = '{first} {last}'.format(
        first=person['first_name'],
        last=person['last_name'],
    )
    return html.format_html('{text} ({login})'.format(
        text=text,
        login=person['login'],
    ))


@register.simple_tag(takes_context=True)
def get_flag_changed_text(context, flag_value):
    language = context['language']
    tpl_data = {
        'ru': ('флаг {action}', 'добавлен', 'снят'),
        'en': ('flag {action}', 'added', 'removed'),
    }[language]
    text_tpl, added_text, removed_text = tpl_data
    if flag_value:
        return text_tpl.format(action=added_text)
    else:
        return text_tpl.format(action=removed_text)


@register.simple_tag(takes_context=True)
def review_action_at_link(context, review, person, text=None):
    language = context['language']
    href = '{url}/reviews/{review_id}?action_at={login}'.format(
        url=settings.FRONTEND_URL,
        review_id=review['id'],
        login=person['login'],
    )

    text = text or {
        'ru': 'Перейти к списку сотрудников',
        'en': 'Go to person list',
    }[language]
    return _make_link(
        href=href,
        text=text,
    )


@register.simple_tag
def review_action_at_href(review, person, statuses):
    statuses = statuses.split(',')
    full_url = '{url}/reviews/{review_id}?action_at={login}'.format(
        url=settings.FRONTEND_URL,
        review_id=review['id'],
        login=person['login'],
    )
    full_url += '&' + '&'.join([
        'status={}'.format(status) for status in statuses
    ])
    return full_url


@register.simple_tag
def genderize(person, text_male, text_female):
    is_male = person['gender'] == staff_const.GENDER.MALE
    if is_male:
        return text_male
    else:
        return text_female


def _make_link(href, text):
    return html.format_html(
        '<a href="{href}">{link_text}</a>',
        href=href,
        link_text=text,
    )


def _num(n, base, post_1, post_2_4, post_5_0):
    """
    Method from django russian lib which not support python 3
    """
    dzat = (((n // 10) % 10) == 1)

    if dzat or n % 10 in [5, 6, 7, 8, 9, 0]:
        return '%s%s' % (base, post_5_0)
    elif n % 10 in [2, 3, 4]:
        return '%s%s' % (base, post_2_4)
    elif n % 10 == 1:
        return '%s%s' % (base, post_1)


def _n_num(n, base, post_1, post_2_4, post_5_0):
    """
    Method from django russian lib which not support python 3
    """
    return '%i %s' % (n, _num(n, base, post_1, post_2_4, post_5_0))


@register.simple_tag
def pluralize_ru(number, base='', one='', two='', five=''):
    return _num(number, base, one, two, five)


@register.simple_tag
def pluralize_ru_with_num(number, base='', one='', two='', five=''):
    return _n_num(number, base, one, two, five)


@register.filter(name='pluralize_ru')
def pluralize_ru_filter(value, arg):
    forms = arg.split(',')
    if len(forms) == 1:
        one = forms[0]
        two, five = ''
    elif len(forms) == 2:
        one, two = forms
        five = ''
    elif len(forms) == 3:
        one, two, five = forms
    else:
        raise Exception('Expected 1 to 3 comma separated args')

    return pluralize_ru(
        number=value,
        base='',
        one=one,
        two=two,
        five=five,
    )


@register.filter
def date_from_str(value, arg):
    """
    Same as django |date filter, but works with string dates
    """
    if not value:
        return ''
    value = arrow.get(value)
    return defaultfilters.date(value, arg)
