# -*- coding: utf-8 -*-

import json
import itertools

from django.contrib import admin
from django import forms
from django.conf import settings
from django.db import transaction

from review.oebs.sync import store
from review.oebs import models
from review.oebs import logic
from review.oebs import const


def copy_to_mock(modeladmin, request, queryset):
    raw_data = logic.get_raw_finance_data(
        person_id__in=queryset.values_list('person_id', flat=True),
    )
    for person_id, finance_data in raw_data.items():
        for key, value in finance_data.items():
            finance_data[key] = json.dumps(
                json.loads(value), ensure_ascii=False)
        models.FinanceMock.objects.update_or_create(
            person_id=person_id,
            defaults=finance_data,
        )
copy_to_mock.short_description = "Скопировать в мок"


class FinanceAdmin(admin.ModelAdmin):
    list_display = (
        'person_login',
    )
    list_select_related = (
        'person',
    )
    list_display_links = None
    search_fields = (
        'person__login',
    )
    raw_id_fields = ('person',)

    change_form_template = None

    actions = [
        copy_to_mock,
    ]

    list_per_page = 10

    ordering = None

    show_full_result_count = False

    def person_login(self, obj):
        return obj.person.login

    def get_queryset(self, request):
        queryset = super(FinanceAdmin, self).get_queryset(request)
        return queryset.only('person__login')


class FinanceMockAdminForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super(FinanceMockAdminForm, self).__init__(*args, **kwargs)
        for name, field in self.fields.items():
            field.required = False

    def save(self, *args, **kwargs):
        for field in const.FINANCE_DB_FIELDS:
            value = getattr(self.instance, field)
            if value:
                value = json.dumps(json.loads(value))
            setattr(self.instance, field, value)
        return super(FinanceMockAdminForm, self).save(*args, **kwargs)

    class Meta:
        model = models.FinanceMock
        fields = '__all__'


def apply_mock(modeladmin, request, queryset):
    store.sync_finance_data(
        person_filter={
            'id__in': queryset.values_list('person_id', flat=True),
        },
    )

apply_mock.short_description = "Применить мок"


class FinanceMockAdmin(admin.ModelAdmin):

    list_display = (
        'person_login',
        'created_at_auto',
        'updated_at_auto',
    )
    list_filter = ('created_at_auto', 'updated_at_auto')

    raw_id_fields = ('person',)

    actions = [
        apply_mock,
    ]

    def person_login(self, obj):
        return obj.person.login

    def get_form(self, request, obj=None, **kwargs):
        if obj is not None:
            for field in const.FINANCE_DB_FIELDS:
                value = getattr(obj, field)
                if value:
                    value = json.dumps(json.loads(value), indent=4)
                setattr(obj, field, value)
        kwargs['form'] = FinanceMockAdminForm
        return super(FinanceMockAdmin, self).get_form(request, obj, **kwargs)

if not settings.IS_PRODUCTION_DB:
    admin.site.register(models.FinanceMock, FinanceMockAdmin)
    admin.site.register(models.Finance, FinanceAdmin)


@transaction.atomic
def reset_currencies_to_default(modeladmin=None, request=None, queryset=None):
    models.Currency.objects.all().delete()
    sorted_defaults = sorted(
        const.DEFAULT_CURRENCIES.ALL,
        key=lambda cur: (cur != const.DEFAULT_CURRENCIES.RUB, cur)
    )
    to_create = (
        models.Currency(
            key=key,
            order=i,
        )
        for i, key in enumerate(sorted_defaults)
    )
    models.Currency.objects.bulk_create(to_create)


reset_currencies_to_default.short_description = "Удалить ВСЕ и залить дефолтные"


@admin.register(models.Currency)
class CurrencyModelAdmin(admin.ModelAdmin):
    list_display_links = (
        'id',
    )
    list_display = (
        'id',
        'key',
        'order',
    )
    list_editable = (
        'key',
        'order',
    )

    actions = [
        reset_currencies_to_default,
    ]


@transaction.atomic
def reset_professions_to_default(modeladmin=None, request=None, queryset=None):
    models.Profession.objects.all().delete()
    order_gen = itertools.count()
    to_create = (
        models.Profession(
            key=key,
            name_ru=name_ru,
            name_en=name_en,
            order=next(order_gen),
        )
        for key, (name_ru, name_en) in sorted(const.DEFAULT_PROFESSIONS_NAMES.items())
    )
    models.Profession.objects.bulk_create(to_create)


reset_professions_to_default.short_description = "Удалить ВСЕ и залить дефолтные"


@admin.register(models.Profession)
class ProfessionModelAdmin(admin.ModelAdmin):
    list_display_links = (
        'id',
    )
    list_display = (
        'id',
        'key',
        'name_ru',
        'name_en',
        'order',
    )
    list_editable = (
        'key',
        'name_ru',
        'name_en',
        'order',
    )

    actions = [
        reset_professions_to_default,
    ]
