# coding: utf-8

import json
import logging

import more_itertools

from review.lib import helpers
from review.lib import encryption
from review.oebs import models as oebs_models
from review.oebs import const

logger = logging.getLogger(__name__)


@helpers.timeit
def get_raw_finance_data(fields=None, group_by='person_id', **filter_condition):
    """
    Маппинг `dict_key` на данные в виде диктов с полями из
    const.FINANCE_DB_FIELDS, в которых лежат расшифрованные строки.
    """
    if fields is None:
        fields = const.FINANCE_DB_FIELDS
    else:
        fields = set(fields)
    fields |= {group_by}
    queryset = oebs_models.Finance.objects.filter(
        **filter_condition
    ).values(*fields)
    decrypted_data = [decrypted_finance_data(item) for item in queryset]
    return {item.pop(group_by): item for item in decrypted_data}


def get_raw_finance_data_from_oebs(fields, person_id__in=None, **unsupported_filters):
    from review.oebs.sync import store
    login_to_id_map = store.get_existing_person_login_to_ids(person_filter={
        'id__in': person_id__in,
    })
    data_for_persons = {}
    for logins_chunk in more_itertools.chunked(list(login_to_id_map.keys()), store.CHUNK_SIZE):
        data_for_persons.update(store.get_api_data(
            logins=logins_chunk,
            login_to_id_map=login_to_id_map,
            data_types=fields,
        ))
    return data_for_persons


def get_finance_data(fields=None, group_by='person_id', cached_result=True, **filter_condition):
    """
    By default fetches data from review db
    If "cached_result=False" is passed - fetches data from oebs api
    """
    if cached_result:
        raw_data = get_raw_finance_data(fields, group_by=group_by, **filter_condition)
        for finance_data in list(raw_data.values()):
            for field, raw_value in finance_data.items():
                data = None
                try:
                    data = json.loads(raw_value)
                except (json.decoder.JSONDecodeError, TypeError):
                    logger.warning("Can't parse {value} in {field}".format(value=raw_value, field=field))

                finance_data[field] = data
    else:
        raw_data = get_raw_finance_data_from_oebs(fields, **filter_condition)
    return raw_data


def get_person_finance_data(login):
    return get_finance_data(
        person__login=login,
        group_by='person__login'
    ).get(login)


def decrypt_finance_data(data, fields=None, deserialize=False):
    """inplace"""
    fields = fields or const.FINANCE_DB_FIELDS
    for key in fields:
        value = data.get(key)
        if not value:
            continue
        decrypted = encryption.decrypt(value)
        if deserialize:
            data[key] = json.loads(decrypted)
        else:
            data[key] = decrypted


def encrypt_finance_data(data, fields=None, serialize=False):
    """inplace"""
    fields = fields or const.FINANCE_DB_FIELDS
    for key in fields:
        value = data.get(key)
        if value is None:
            continue
        encrypted = encryption.encrypt(value)
        if serialize:
            data[key] = json.dumps(encrypted)
        else:
            data[key] = encrypted


def encrypt_finance_event(event, serialize=True):
    if serialize:
        event = json.dumps(event)
    return encryption.encrypt(event)


def decrypt_finance_event(event, deserialize=True):
    decrypted = encryption.decrypt(event)
    if deserialize:
        decrypted = json.loads(decrypted)
    return decrypted


def decrypted_finance_data(data, fields=None, deserialize=False):
    """make copy"""
    result = dict(data)
    decrypt_finance_data(result, fields, deserialize)
    return result


def encrypted_finance_data(data, fields=None, serialize=False):
    """make copy"""
    result = dict(data)
    encrypt_finance_data(result, fields, serialize)
    return result


def stringify_values(data, keep_errors=False):
    result = {}
    for key, value in data.items():
        if key not in const.FINANCE_DB_FIELDS:
            result[key] = value
            continue
        if keep_errors and value == const.OEBS_DATA_FETCH_ERROR:
            result[key] = value
            continue
        result[key] = json.dumps(value)
    return result
