# coding: utf-8
import datetime
import json

import arrow
from cached_property import cached_property

from review.lib import datetimes
from review.lib import helpers
from review.staff import models as staff_models
from review.oebs import const
from review.oebs import models as oebs_models


CURRENCIES = const.DEFAULT_CURRENCIES

CITY_TO_CURRENCY = {
    'Ньюберипорт': CURRENCIES.USD,
    'Киев': CURRENCIES.UAH,
    'Одесса': CURRENCIES.UAH,
    'Амстердам': CURRENCIES.EUR,
    'Хельсинки': CURRENCIES.EUR,
    'Люцерн': CURRENCIES.CHF,
    'Шанхай': CURRENCIES.CNY,
    'Тель-Авив': CURRENCIES.ILS,
}


@helpers.timeit
def generate_data(data_types, logins):
    persons_data = staff_models.Person.objects.filter(login__in=logins).values(
        'id',
        'login',
        'join_at',
        'city_name_ru',
    )

    mock_data = oebs_models.FinanceMock.objects.filter(
        person__login__in=logins
    ).in_bulk()

    oebs_data = {}
    for person_data in persons_data:
        all_person_data = oebs_data.setdefault(person_data['login'], {})
        for data_type in data_types:
            method = {
                const.SALARY_HISTORY: generate_salary_history,
                const.GRADE_HISTORY: generate_grade_history,
                const.BONUS_HISTORY: generate_bonus_history,
                const.OPTION_HISTORY: generate_options_history,
                const.CURRENT_LOANS: generate_current_loans,
                const.CURRENT_SALARY: generate_current_salary,
                const.SOCIAL_PACKAGE: generate_social_package,
            }.get(data_type)

            mock_value = mock_data.get(person_data['id'])
            mock_value = mock_value and getattr(mock_value, data_type)

            if mock_value:
                person_oebs_data = json.loads(mock_value)
            else:
                faker = Faker(person_data)
                person_oebs_data = method(faker)
            all_person_data[data_type] = person_oebs_data
    return oebs_data


class Faker(object):

    def __init__(self, person_data):
        for key, value in person_data.items():
            setattr(self, key, value)

    @property
    def currency(self):
        return CITY_TO_CURRENCY.get(self.city_name_ru, 'RUB')

    @property
    def join_at_year(self):
        return self.join_at.year

    @property
    def initial_salary(self):
        current_year = arrow.now().year
        yandex_start_year = 2000
        initial_salary_at_yandex_start = 20000
        initial_salary_year_raise = 5000
        work_salary_year_raise = 15000

        join_year_initial_salary = initial_salary_at_yandex_start + (
            self.join_at_year - yandex_start_year) * initial_salary_year_raise
        return join_year_initial_salary + (
            current_year - self.join_at_year) * work_salary_year_raise

    @cached_property
    def salary_events(self):
        result = []
        raise_in_period = 15000
        dates = generate_dates(
            start_date=self.join_at,
            step_shift={'months': 6}
        )
        for index, date in enumerate(dates):
            salary = self.initial_salary + index * raise_in_period
            result.append((date, salary))
        return result

    @cached_property
    def level_events(self):
        result = []
        raise_in_period = 1
        dates = generate_dates(
            start_date=self.join_at,
            step_shift={'years': 1}
        )
        initial_grade = 2
        for index, date in enumerate(dates):
            salary = initial_grade + index * raise_in_period
            result.append((date, salary))
        return result

    def get_salary_for_date(self, date):
        for salary_date, salary_sum in reversed(self.salary_events):
            if date > salary_date:
                return salary_sum

    @cached_property
    def grade_type(self):
        choices = list(const.DEFAULT_PROFESSIONS)
        return choices[self.id % len(choices)]

    @cached_property
    def grade_speciality(self):
        return self.id % 7 + 1

    @cached_property
    def grade_name(self):
        return '%s.%s.%s' % (
            self.grade_type,
            self.current_grade,
            self.grade_speciality,
        )

    @cached_property
    def bonus_events(self):
        result = []
        bonus_period = self.get_one_of([3, 6])
        bonus_for_period = {
            6: (100, 20),
            3: (50, 10),
        }[bonus_period]
        dates = generate_dates(
            start_date=self.join_at,
            initial_shift={'months': 3},
            step_shift={'months': bonus_period}
        )
        for index, date in enumerate(dates):
            date = datetimes.replaced(date, day=1)
            std_bonus, extra_bonus = bonus_for_period
            bonus_percent = (std_bonus + self.id % extra_bonus) / 100
            bonus_sum = bonus_percent * self.get_salary_for_date(date)
            result.append((date, bonus_sum))
        return result

    @cached_property
    def options_events(self):
        result = []
        dates = generate_dates(
            start_date=min(self.join_at, datetime.date(2015, 1, 1)),
            initial_shift={'months': 6},
            step_shift={'months': 12},
        )
        for index, date in enumerate(dates):
            value = ((self.id + index) % 12 + 1) * 100
            result.append((date, value))
        return result

    @cached_property
    def current_salary(self):
        _, current_salary = self.salary_events[-1]
        return current_salary

    @cached_property
    def current_grade(self):
        _, level_grade = self.level_events[-1]
        return level_grade

    @cached_property
    def current_loans(self):
        result = []
        dates = generate_dates(
            start_date=self.join_at,
            initial_shift={'years': 1},
            step_shift={'years': 4},
        )
        today = arrow.now().date()
        for index, date in enumerate(dates):
            if self.id % 10 != 3:
                continue
            end_date = datetimes.shifted(date, years=+3)
            if today >= end_date:
                continue

            value = self.get_salary_for_date(date) * 36 / 3
            result.append((date, value))
            break
        return result

    @cached_property
    def social_package(self):
        return {
            "food": self.get_yes_or_no('food'),
            "mobile": self.get_yes_or_no('mobile'),
            "vhi": self.get_yes_or_no('vhi'),
            "internet": self.get_yes_or_no('internet'),
        }

    def get_one_of(self, list_, randomizer=''):
        return list_[len(self.login + randomizer) % len(list_)]

    def get_yes_or_no(self, randomizer=''):
        return self.get_one_of(['Y', 'N'], randomizer=randomizer)


def generate_dates(start_date, step_shift, initial_shift=None, end_date=None):
    end_date = end_date or arrow.now().date()
    current = arrow.get(start_date).date()
    if initial_shift is not None:
        current = datetimes.replaced(start_date, **initial_shift)

    yield current
    while current <= end_date:
        current = datetimes.replaced(current, **step_shift)
        yield current


def generate_salary_history(faker):
    history = []
    for date, value in faker.salary_events:
        history.append(
            {
                "hiddenInfo": False,
                "salarySum": value,
                "currency": faker.currency,
                "basis": "MONTHLY",
                "dateFrom": date.isoformat(),
            }
        )
    set_date_to(history)

    return history


def generate_grade_history(faker):
    history = []
    for date, value in faker.level_events:
        history.append(
            {
                "gradeCity": faker.city_name_ru,
                "gradeName": faker.grade_name,
                "dateFrom": date.isoformat(),
                "currency": None,
                "max": None,
                "min": None,
                "mid": None,
            },
        )
    set_date_to(history)

    return history


def generate_bonus_history(faker):
    history = []
    for date, value in faker.bonus_events:
        history.append(
            {
                "elementName": "Премия годовая",
                "dateFrom": date.isoformat(),
                "dateTo": arrow.get(date).ceil('month').date(),
                "effectiveDate": datetimes.shifted(date, days=+20).isoformat(),
                "hiddenInfo": False,
                "cfoName": "Какой-то отдел",
                "cfoNameEng": "Department of something",
                "anotherBudget": faker.get_yes_or_no('anotherBudget'),
                "currency": faker.currency,
                "summ": value,
            },
        )
    set_date_to(history)
    return history


def generate_options_history(faker):
    history = []
    grant_id = 0
    for date, value in faker.options_events:
        grant_id += 1
        vesting = []
        for vest_date in generate_dates(
                start_date=date,
                initial_shift={
                    'months': +3,
                },
                step_shift={
                    'months': +3,
                },
                end_date=datetimes.shifted(date, years=+4),
        ):
            vesting.append(build_vesting(
                value / 16,
                vest_date,
            ))
        history.append(build_grant(vesting, grant_id))
    return history


def build_grant(
    vesting,
    grant_id=0,
    type_='RSU',
    cur_price=65.605,
    price_for_user=0,
    currency='USD',
):
    if not vesting:
        total = 0
        lapse_date = datetime.date.max
    else:
        total = sum(it['vestAmount'] for it in vesting)
        lapse_date = vesting[-1]['vestDate']
    return {
        'shareCostSysdate': cur_price,
        'sourceCreation': 'REVIEW',
        'entityName': 'Yandex NV, The Netherlands',
        'grantPrice': price_for_user,
        'planExercisable': 0,
        'lapseDate': lapse_date,
        'grantExercisableUnits': 1,
        'createFromDeals': 'N',
        'grantAmount': total,
        'grantCurrency': currency,
        'vesting': vesting,
        'classCode': type_,
        'grantId': grant_id,
    }


def build_vesting(amount, date):
    return {
        'vestAmount': amount,
        'vestDate': date.isoformat(),
        'vestMoney': None,
    }


def generate_current_loans(faker):
    history = []
    today = arrow.now().date()
    for date, value in faker.current_loans:
        end_date = datetimes.shifted(date, years=+3)
        balance = ((end_date - today).days / float(365 * 3)) * value
        balance = int(balance * 100) / float(100)
        history.append({
            "contractDate": date.isoformat(),
            "contractNum": date.strftime('%d/%m/%y'),
            "loanType": "20",
            "percent": 0,
            "loanSum": value,
            "currency": "RUB",
            "balance": balance,
        })
    return history


def generate_current_salary(faker):
    return {
        "basis": "MONTHLY",
        "currency": faker.currency,
        "factFTE": 1,
        "gradeCity": faker.city_name_ru,
        "gradeCurrency": None,
        "gradeName": faker.grade_name,
        "hiddenInfo": False,
        "max": None,
        "mid": None,
        "min": None,
        "salarySum": faker.current_salary,
    }


def generate_social_package(faker):
    return faker.social_package


def set_date_to(history):
    date_to = const.MAX_HISTORY_DATE
    for event in reversed(history):
        event['dateTo'] = date_to
        date_to = arrow.get(event['dateFrom']).replace(days=-1).date().isoformat()
