# coding: utf-8

import logging

from django.conf import settings
from ids import exceptions
from ids.connector import HttpConnector, plugins_lib
import yenv

from review.lib import helpers
from review.lib.tvm2_client import get_service_ticket
from review.oebs import const
from review.oebs.sync import fake


log = logging.getLogger(__name__)


class OebsConnector(HttpConnector):
    default_connect_timeout = 20

    url_prefix = '/rest'

    url_patterns = {
        const.SALARY_HISTORY: '/salaryHist',
        const.GRADE_HISTORY: '/gradeHist',
        const.BONUS_HISTORY: '/bonus',
        const.OPTION_HISTORY: '/options',
        const.CURRENT_LOANS: '/loan',
        const.CURRENT_SALARY: '/salary',
        const.SOCIAL_PACKAGE: '/socialPackage',
        const.KPI: '/getPersonAwd',
        const.LOAN_CALC: '/loanCalculation',
    }

    plugins = (
        plugins_lib.JsonResponse,
    )

    def __init__(self, **kwargs):
        protocol = kwargs.pop('protocol', settings.OEBS_PROTOCOL)
        host = kwargs.pop('host', settings.OEBS_HOST)
        retries = kwargs.pop('retries', 5)
        user_agent = kwargs.pop('user_agent', settings.IDS_DEFAULT_USER_AGENT)

        super(OebsConnector, self).__init__(
            host=host,
            protocol=protocol,
            retries=retries,
            user_agent=user_agent,
            **kwargs
        )

        if yenv.type == 'development':
            self.session.verify = False

    def prepare_params(self, params):
        headers = params.get('headers', {})
        headers['X-Ya-Service-Ticket'] = get_service_ticket(settings.TVM_APP_IDS['oebs'])
        params['headers'] = headers

    def _handle_response(self, response):
        if response.ok and not response.content:
            return {}
        return super(OebsConnector, self)._handle_response(response)


# TODO: retries for all errors, maybe split in smaller chunks
def get_raw_oebs_data(data_type, logins):
    connector = OebsConnector()

    try:
        return connector.post(resource=data_type, json=get_request_params(data_type, logins))
    except exceptions.BackendError as exc:
        response_content = exc.response is not None and exc.response.content
        status_code = exc.response is not None and exc.response.status_code
        log.exception(
            'Oebs bad response %s `%s`. data_type=%s, logins=%s',
            status_code,
            response_content,
            data_type,
            logins,
        )
        return const.OEBS_DATA_FETCH_ERROR


def get_request_params(data_type, logins):
    result = {'login': logins}
    if data_type == const.OPTION_HISTORY:
        result.update({'showGrants': {
            'declared': 'Y',
            'grantStatus': ['AGREED', 'PRE_AGREED', 'NEW', 'UPLOADED'],
        }})
    return result


def normalize_raw_data(data_type, raw_data):
    # в выдаче всегда есть разный бесполезный ключ верхнего уровня
    data = raw_data.get(const.ROOT_COLLECTION_NAMES[data_type], [])
    data_by_login = {}
    for datum in data:
        # если сущность несписковая, то логин лежит внутри, иначе — рядом
        if data_type in const.NON_LIST_DATA_TYPES:
            login = datum.pop('login')
            real_data = datum
        else:
            login = datum['login']
            real_data = datum[const.INNER_COLLECTION_NAMES[data_type]]
        if login not in data_by_login:
            data_by_login[login] = real_data
        else:
            # один и тот же логин может прийти несколько раз
            existing_data = data_by_login[login]
            if isinstance(existing_data, list):
                existing_data += real_data
            elif isinstance(existing_data, dict):
                # поведение не определено, будем использовать последний пришедший
                data_by_login[login] = real_data
    return data_by_login


@helpers.timeit_no_args_logging
def get_oebs_data_real(data_types, logins):
    result = {}
    for data_type in data_types:
        raw_data = get_raw_oebs_data(data_type, logins)

        if raw_data == const.OEBS_DATA_FETCH_ERROR:
            for login in logins:
                person_data = result.setdefault(login, {})
                person_data[data_type] = const.OEBS_DATA_FETCH_ERROR
            continue

        normalized = normalize_raw_data(data_type, raw_data)
        for login in logins:
            if login in normalized:
                value_for_type = normalized[login]
            elif data_type in const.NON_LIST_DATA_TYPES:
                value_for_type = {}
            else:
                value_for_type = []
            person_data = result.setdefault(login, {})
            person_data[data_type] = value_for_type
    return result


@helpers.timeit_no_args_logging
def get_oebs_data_fake(data_types, logins):
    return fake.generate_data(data_types, logins)


default_sync = yenv.choose_key_by_type(
    dict_={
        'testing': get_oebs_data_fake,
        'production': get_oebs_data_real,
    },
    fallback=True,
)


def get_oebs_data(data_types, logins, sync_type=None):
    if sync_type is None or isinstance(sync_type, str):
        fu = {
            'real': get_oebs_data_real,
            'fake': get_oebs_data_fake,
        }.get(sync_type, default_sync)
        return fu(data_types, logins)
    else:
        real_data = get_oebs_data_real(sync_type['real_types'], logins)
        fake_data = get_oebs_data_fake(sync_type['fake_types'], logins)
        for k, v in real_data.items():
            v.update(fake_data[k])
        return real_data
