# coding: utf-8

import logging
import arrow
import more_itertools
from bulk_update.helper import bulk_update
from django.conf import settings
from django.db import transaction
from ylog.context import log_context

from review.lib import helpers
from review.oebs import const
from review.oebs import logic as oebs_logic
from review.oebs import models as oebs_models
from review.oebs.sync import fetch
from review.staff import models as staff_models

CHUNK_SIZE = settings.SYNC_FINANCE_DATA_CHUNK_SIZE

log = logging.getLogger(__name__)


@helpers.timeit_no_args_logging
def sync_finance_data(
    chunk_size=CHUNK_SIZE,
    person_filter=None,
    sync_type=None,
    data_types=const.OEBS_DATA_TYPES,
):
    """
    sync type can be:
    1. None
    2. "fake" or "real" - to run sync data fully from oebs or mocks
    3. dictionary with two keys "fake" and "real". Values have to be lists,
       contains OEBS_TYPES to sync. Example:
       {
           "fake_types": [
                const.SALARY_HISTORY,
                const.GRADE_HISTORY,
                const.BONUS_HISTORY,
                const.CURRENT_SALARY,
                const.CURRENT_LOANS,
                const.SOCIAL_PACKAGE
           ],
           "real_types": [
                const.OPTION_HISTORY,
           ]
       }
    """
    updated = created = fetched = 0
    login_to_id_map = get_existing_person_login_to_ids(person_filter=person_filter)
    for logins_chunk in more_itertools.chunked(list(login_to_id_map.keys()), chunk_size):
        api_data = get_api_data(
            logins=logins_chunk,
            login_to_id_map=login_to_id_map,
            data_types=data_types,
            sync_type=sync_type,
        )
        db_data = get_db_data(
            person_ids=[login_to_id_map[login] for login in logins_chunk]
        )
        ids_without_finance_in_db = set(api_data) - set(db_data)
        to_create = {
            person_id: api_data[person_id]
            for person_id in ids_without_finance_in_db
        }
        created += create_finance_data_bulk(api_data=to_create)
        local_num = len(db_data) + len(to_create)
        if len(api_data) != local_num:
            logging.error('api returned not all logins: received {}, expected {}'.format(
                len(api_data),
                local_num,
            ))
        updated += update_db_data_if_changed(
            db_data=db_data,
            api_data={person_id: api_data[person_id] for person_id in db_data}
        )
        if const.SALARY_HISTORY in data_types:
            update_salary_events(api_data)
        if const.GRADE_HISTORY in data_types:
            update_grade_events(api_data)
        fetched += len(logins_chunk)
    return {
        'created': created,
        'updated': updated,
        'fetched': fetched,
    }


@helpers.timeit_no_args_logging
def get_existing_person_login_to_ids(person_filter=None):
    queryset = staff_models.Person.objects
    filters = {}
    if person_filter is not None:
        filters.update(person_filter)
    queryset = queryset.filter(**filters)
    return dict(queryset.values_list('login', 'id'))


@helpers.timeit_no_args_logging
def get_api_data(logins, login_to_id_map, data_types=const.OEBS_DATA_TYPES, sync_type=None):
    all_data = fetch.get_oebs_data(
        data_types=data_types,
        logins=logins,
        sync_type=sync_type,
    )
    return {
        login_to_id_map[login]: person_data
        for login, person_data in all_data.items()
    }


@helpers.timeit_no_args_logging
def get_db_data(person_ids):
    return oebs_logic.get_raw_finance_data(person_id__in=person_ids)


@helpers.timeit_no_args_logging
def create_finance_data_bulk(api_data):
    models = []
    for person_id, data in api_data.items():
        # даже если каких-то полей нет — при создании положим в базу пустой
        # объект, как будто все ок
        data = oebs_logic.stringify_values(data, keep_errors=False)
        encrypted = oebs_logic.encrypted_finance_data(data)
        model = oebs_models.Finance(
            person_id=person_id,
            **{
                key: value for key, value in encrypted.items()
                if key in const.FINANCE_DB_FIELDS
            }
        )
        models.append(model)
    return len(oebs_models.Finance.objects.bulk_create(models))


@helpers.timeit_no_args_logging
def update_salary_events(api_data):
    update_event_model(
        api_data=api_data,
        date_from_field='dateFrom',
        date_to_field='dateTo',
        target_data_type=const.SALARY_HISTORY,
    )


@helpers.timeit_no_args_logging
def update_grade_events(api_data):
    update_event_model(
        api_data=api_data,
        date_from_field='dateFrom',
        date_to_field='dateTo',
        target_data_type=const.GRADE_HISTORY,
    )


def update_event_model(api_data, date_from_field, date_to_field, target_data_type):
    persons_api_events = {}
    for person_id, person_data in api_data.items():
        for data_type, events in person_data.items():
            if data_type == target_data_type:
                for event in events:
                    try:
                        date_from = arrow.get(event.pop(date_from_field)).date()
                        date_to = arrow.get(event.pop(date_to_field)).date()
                        persons_api_events[(person_id, date_from, date_to)] = event
                    except Exception:
                        log.warning(f'Can\'t parse data for {person_id}')
                        raise

    person_ids = list(api_data.keys())
    db_type = const.FINANCE_EVENT_TYPES[target_data_type]
    db_data = oebs_models.FinanceEvents.objects.filter(
        person_id__in=person_ids,
        type=db_type,
    ).values('id', 'person_id', 'date_from', 'date_to', 'event')
    persons_db_events = {
        (item['person_id'], item['date_from'], item['date_to']): item['event']
        for item in db_data
    }
    persons_db_event_ids = {
        (item['person_id'], item['date_from'], item['date_to']): item['id']
        for item in db_data
    }

    to_delete = persons_db_event_ids.keys() - persons_api_events.keys()
    to_delete = {event_id for key, event_id in persons_db_event_ids.items() if key in to_delete}
    to_create = persons_api_events.keys() - persons_db_events.keys()
    to_create = {
        key: oebs_logic.encrypt_finance_event(event)
        for key, event in persons_api_events.items()
        if key in to_create
    }

    to_update = []
    for key in persons_db_event_ids.keys() & persons_api_events.keys():
        api_event = persons_api_events[key]
        db_event = oebs_logic.decrypt_finance_event(persons_db_events[key])
        if not set(api_event.items()) ^ set(db_event.items()):
            continue
        encrypted_event = oebs_logic.encrypt_finance_event(api_event)
        to_update.append(oebs_models.FinanceEvents(
            id=persons_db_event_ids[key],
            event=encrypted_event,
        ))

    to_create = [
        oebs_models.FinanceEvents(
            person_id=person_id,
            date_from=date_from,
            date_to=date_to,
            event=event,
            type=db_type,
        )
        for (person_id, date_from, date_to), event in to_create.items()
    ]
    with transaction.atomic():
        oebs_models.FinanceEvents.objects.filter(id__in=to_delete).delete()
        oebs_models.FinanceEvents.objects.bulk_create(to_create)
        bulk_update(to_update, update_fields=['event'])


@helpers.timeit_no_args_logging
def update_db_data_if_changed(db_data, api_data):
    updated_count = 0
    for person_id, api_item in api_data.items():
        with log_context(person_id=person_id):
            db_item = db_data[person_id]
            is_updated = update_one_model_if_changed(
                person_id=person_id,
                db_item=db_item,
                api_item=api_item
            )
            if is_updated:
                updated_count += 1
    return updated_count


def update_one_model_if_changed(person_id, db_item, api_item):
    diff = get_finance_data_diff(db_item, api_item)
    if not diff:
        return False

    log.debug('Finance for id=%d needs update', person_id)

    fields_for_update = {}
    for db_field, old_val, new_val in diff:
        fields_for_update[db_field] = new_val

    fields_for_update = oebs_logic.encrypted_finance_data(fields_for_update)
    oebs_models.Finance.objects.filter(
        person_id=person_id
    ).update(**fields_for_update)
    return True


def get_finance_data_diff(db_item, api_item):
    diff = []
    api_item = oebs_logic.stringify_values(api_item, keep_errors=True)
    for db_field in set(const.FINANCE_DB_FIELDS):
        if db_field not in api_item:
            log.debug('Skip updating `%s` because it is not present in api_item', db_field)
            continue

        old_value = db_item[db_field]
        new_value = api_item[db_field]
        # не затираем хорошие данные полученными с ошибкой
        if new_value == const.OEBS_DATA_FETCH_ERROR:
            log.debug('Skip updating `%s` because of OEBS_DATA_FETCH_ERROR', db_field)
            continue
        if old_value == new_value:
            log.debug('Skip updating `%s` because it was not changed', db_field)
            continue
        diff.append((db_field, old_value, new_value))
    return diff
