# -*- coding: utf-8 -*-
from django.contrib import admin
from django.contrib.admin.sites import site
from django.contrib.sites import models as sites_models
from django.conf import settings

from treebeard import admin as tree_admin

from . import models

from review.core import const as core_const
from review.lib.admin import ReadOnlyProductionModelAdmin, ReadOnlyProductionAdminMixin


site.unregister(sites_models.Site)


@admin.register(models.Person)
class PersonAdmin(admin.ModelAdmin):
    list_per_page = 20

    fields = (
        'uid',
        'login',
        'department',
        'is_dismissed',
        'is_robot',
        'gender',
        'position_ru',
        'join_at',
        'work_email',
        'language',
        'groups',
    )
    list_display = (
        'id',
        'global_role',
        'uid',
        'login',
        'department',
        'is_dismissed',
        'is_robot',
        'gender',
        'position_ru',
        'join_at',
        'work_email',
        'language',
    )
    readonly_fields = (
        'uid',
        'login',
        'department',
        'is_dismissed',
        'is_robot',
        'gender',
        'position_ru',
        'join_at',
        'work_email',
        'language',
    )

    list_filter = (
        'is_dismissed',
        'is_robot',
        'join_at',
    )

    search_fields = (
        'login',
        'id',
        'uid',
        'department__slug',
    )

    raw_id_fields = ('department',)

    list_select_related = (
        'department',
    )
    actions = []

    def get_queryset(self, request):
        queryset = super(PersonAdmin, self).get_queryset(request)
        return queryset.prefetch_related('global_roles')

    def global_role(self, instance):
        roles = instance.global_roles.all()
        if not roles:
            return None
        else:
            return ', '.join(core_const.ROLE.GLOBAL.VERBOSE[role.type] for role in roles)

    def has_delete_permission(self, request, obj=None):
        return not settings.IS_PRODUCTION_DB

    def has_add_permission(self, request):
        return not settings.IS_PRODUCTION_DB


@admin.register(models.Department)
class DepartmentAdmin(ReadOnlyProductionAdminMixin, tree_admin.TreeAdmin):
    fields = (
        'slug',
        'path',
        'depth',
        'numchild',
        'name_ru',
    )
    list_display = fields + ('id', 'heads',)
    actions = []

    search_fields = (
        'slug',
        'path',
        'name_ru',
        'id',
    )

    def move_node(self, request):
        pass

    def get_queryset(self, request):
        queryset = super(DepartmentAdmin, self).get_queryset(request)
        return queryset.prefetch_related('roles', 'roles__person')

    def heads(self, instance):
        roles = instance.roles.all()
        if not roles:
            return None
        else:
            return ', '.join(role.person.login for role in roles)


class DepartmentRoleAdmin(admin.ModelAdmin):
    fields = (
        'person',
        'department',
        'type'
    )
    list_display = ('id',) + fields
    readonly_fields = () if not settings.IS_PRODUCTION_DB else fields

    list_select_related = (
        'person',
        'department',
    )
    list_filter = ('type',)
    search_fields = (
        'person__login',
        'department__slug',
    )
    raw_id_fields = ('person', 'department')

    def has_delete_permission(self, request, obj=None):
        return True

    def has_add_permission(self, request):
        return True


@admin.register(models.Subordination)
class SubordinationAdmin(ReadOnlyProductionModelAdmin):
    fields = list_display = (
        'subject',
        'object',
        'type',
        'position',
    )
    list_display = ('id', ) + fields
    search_fields = ('subject__login', 'object__login')


@admin.register(models.HR)
class HRAdmin(ReadOnlyProductionModelAdmin):
    fields = list_display = (
        'cared_person',
        'hr_person',
        'type',
    )
    list_display = ('id', ) + fields
    search_fields = ('cared_person__login', 'hr_person__login')


@admin.register(models.StaffStructureChange)
class StaffStructureChangeAdmin(ReadOnlyProductionModelAdmin):
    fields = list_display = (
        'date',
        'staff_id',
        'processed_at',
    )
    list_display = ('id', ) + fields
    search_fields = ('date', )


@admin.register(models.PersonHeads)
class PersonHeadsAdmin(ReadOnlyProductionModelAdmin):
    fields = list_display = (
        'structure_change',
        'person_id',
        'heads',
    )
    list_display = ('id', ) + fields
    search_fields = ('person_id', )


admin.site.register(models.DepartmentRole, DepartmentRoleAdmin)

admin.site.disable_action('delete_selected')
