# coding: utf-8

import six

from django.db.models import Q

from review.staff import models
from review.staff import const
from typing import List, AnyStr, Dict


# public API
# Руководимые объекты (подразделения и люди руководителя)
def get_direct_subordinate_departments(person):
    """
    Прямые подразделения под руководством `person`.
    Бывают вложенными в друг друга.
    """
    person = ensure_person_model(person)
    return models.Department.objects.filter(
        role__person=person,
        role__type=const.STAFF_ROLE.DEPARTMENT.HEAD,
    )


def get_all_subordinate_departments(person):
    """
    Все подразделения под руководством `person`.
    """
    return models.Department.objects.filter(
        get_all_subordinate_departments_q(person)
    )


def get_all_subordinate_persons(person):
    """
    Все сотрудники из всех подразделений, руководимых `person`.
    """
    return models.Person.objects.filter(
        get_all_subordinate_persons_q(person)
    )


# Руководящие объекты (вышестоящие подразделения и их руководители)
def get_person_heads(person):
    """
    Цепочка руководителей сотрудника `person`.
    """
    person = ensure_person_model(person)
    head_roles = models.DepartmentRole.objects.filter(
        get_department_ancestors_q(
            department=person.department,
            include_self=True,
            prefix='department__',
        ),
        type=const.STAFF_ROLE.DEPARTMENT.HEAD,
    ).select_related('person').order_by('-department__depth')
    return [role.person for role in head_roles]


def get_department_head(department):
    department = ensure_department_model(department)
    head_role = department.roles.filter(
        type=const.STAFF_ROLE.DEPARTMENT.HEAD
    ).select_related('person').first()
    return head_role and head_role.person


def get_person_hrs(person):
    """
    Множество эйчаров сотрудника из его цепочки.
    """
    person = ensure_person_model(person)
    hr_roles = models.DepartmentRole.objects.filter(
        get_department_ancestors_q(
            department=person.department,
            include_self=True,
            prefix='department__',
        ),
        type__in=(const.STAFF_ROLE.HR.HR_ANALYST, const.STAFF_ROLE.HR.HR_PARTNER),
    ).select_related('person')
    return [(role.person, role.type) for role in hr_roles]


# включая/исключая
# def get_head_departments(person):
#     person = ensure_person_model(person)
#     return person.department.get_ancestors()


# Сотрудники/подразделения без привязки к руководству
def get_persons_from_department_tree(root):
    """
    Все сотрудники в дереве подразделений,
    корнем которого является `department`.
    """
    ensure_department_model(root)
    return models.Person.objects.filter(
        get_persons_from_department_tree_q(
            root=root,
            include_root=True,
        )
    )


def get_persons_from_department_forest(roots):
    """
    Все сотрудники в дереве подразделений,
    корнем которого является `department`.
    """
    return models.Person.objects.filter(
        get_persons_from_department_forest_q(
            roots=roots,
            include_root=True,
        )
    )


# Q-builders
def get_filter_nothing_q(prefix):
    return Q(**{prefix + 'id__isnull': True})


def get_all_subordinate_persons_q_via_deps(person, prefix=''):
    person = ensure_person_model(person)
    departments_filter = get_all_subordinate_departments_q(
        person=person,
        prefix=prefix + 'department__',
    )
    exlude_self_filter = ~Q(**{prefix + 'id': person.id})
    dismissal_filter = Q(**{prefix + 'is_dismissed': False})
    return departments_filter & dismissal_filter & exlude_self_filter


def get_all_subordinate_persons_q_via_subordination(person, prefix=''):
    person = ensure_person_model(person)
    subordination_filter = Q(**{
        prefix + 'superior__subject': person,
    })
    exlude_self_filter = ~Q(**{prefix + 'id': person.id})
    return subordination_filter & exlude_self_filter


get_all_subordinate_persons_q = get_all_subordinate_persons_q_via_subordination


def get_chief_to_empl_id(chief_ids):
    return (
        models.Subordination.objects
        .filter(subject_id__in=chief_ids)
        .values_list('subject_id', 'object_id')
    )


def get_all_subordinate_departments_q(person, prefix=''):
    departments = get_direct_subordinate_departments(person)
    return get_department_flat_forest_q(
        roots=departments,
        prefix=prefix,
    )


def get_persons_from_department_tree_q(root, prefix='', include_root=True):
    return get_persons_from_department_forest_q(
        roots=[root],
        prefix=prefix,
        include_root=include_root,
    )


def get_persons_from_department_forest_q(roots, prefix='', include_root=True):
    query = get_filter_nothing_q(prefix)
    for root_department in roots:
        query |= get_department_flat_tree_q(
            root=root_department,
            prefix=prefix + 'department__',
            include_root=include_root,
        )
    return query


def get_persons_from_department_root_q(root_path, prefix=''):
    return Q(**{prefix + 'department__path__startswith': root_path})


def get_department_flat_tree_q(root, prefix='', include_root=True):
    root = ensure_department_model(root)
    q_params = {prefix + 'path__startswith': root.path}
    if not include_root:
        q_params[prefix + 'depth__gt'] = root.depth
    return Q(**q_params)


def get_department_flat_forest_q(roots, prefix='', include_root=True):
    query = get_filter_nothing_q(prefix)
    for root in roots:
        query |= get_department_flat_tree_q(root, prefix, include_root)
    return query


def get_department_ancestors_q(department, include_self=False, prefix=''):
    return Q(**{
        prefix + 'path__in': _get_ancestors_paths(department, include_self)
    })


def _get_ancestors_paths(department, include_self=False):
    return models.Department.get_ancestor_paths(
        path=department.path,
        include_self=include_self,
    )


# utils
def ensure_department_model(department):
    if isinstance(department, int):
        return models.Department.objects.get(id=department)
    elif isinstance(department, six.string_types):
        return models.Department.objects.get(slug=department)
    return department


def ensure_person_model(person):
    if isinstance(person, int):
        return models.Person.objects.get(id=person)
    elif isinstance(person, six.string_types):
        return models.Person.objects.get(login=person)
    return person


def get_is_chief_by_login(person_login):
    return models.Subordination.objects.filter(subject__login=person_login).exists()


def get_hr_roles_by_login(person_login):
    return models.HR.objects.filter(hr_person__login=person_login).values_list('type', flat=True).distinct()
