# coding: utf-8
from django.core import validators
from django.db import models
from django.contrib.auth import models as auth_models
from treebeard import mp_tree
import yenv

from . import const
from review.core import const as core_const


class Person(auth_models.AbstractBaseUser, auth_models.PermissionsMixin):
    uid = models.BigIntegerField(
        unique=True,
    )
    login = models.CharField(
        max_length=50,
        unique=True,
    )

    department = models.ForeignKey(
        to='staff.Department',
        related_name='persons',
        related_query_name='person',
        # for sync period
        on_delete=models.SET_NULL,
        null=True,
        default=None,
    )

    is_dismissed = models.BooleanField(
        default=False,
        db_index=True,
    )
    is_robot = models.BooleanField(
        default=False,
        db_index=True,
    )
    gender = models.CharField(
        max_length=1,
        choices=const.GENDER.CHOICES,
        default=const.GENDER.MALE,
    )

    marks_format_mode = models.SmallIntegerField(default=1)

    first_name_ru = models.TextField(default="")
    last_name_ru = models.TextField(default="")
    first_name_en = models.TextField(default="")
    last_name_en = models.TextField(default="")

    city_name_ru = models.TextField(default="")
    city_name_en = models.TextField(default="")

    position_ru = models.TextField(default="")
    position_en = models.TextField(default="")
    join_at = models.DateField(null=True, default=None)
    quit_at = models.DateField(null=True, default=None)

    work_email = models.CharField(max_length=100, default="")
    language = models.CharField(max_length=10, default="ru")
    timezone = models.CharField(max_length=50, default="Europe/Moscow")

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    # custom user model stuff

    USERNAME_FIELD = 'login'
    EMAIL_FIELD = 'work_email'

    def get_short_name(self):
        return '%s %s' % (self.first_name_ru, self.last_name_ru)

    def get_full_name(self):
        return self.get_short_name()

    @property
    def is_active(self):
        return not self.is_dismissed

    @property
    def is_support(self):
        return self.global_roles.filter(type__exact=core_const.ROLE.GLOBAL.SUPPORT).exists()

    @property
    def is_compensations_user(self):
        # Temporary method for Compensations MVP.
        # Should be deleted after moving compensations app to separate service.
        return self.has_module_perms('compensations')

    @property
    def is_staff(self):
        return self.is_support or self.is_compensations_user

    @property
    def is_superuser(self):
        if yenv.type != 'production':
            return self.is_support

    def has_perm(self, perm, obj=None):
        return self.is_support or super().has_perm(perm, obj)

    def has_module_perms(self, app_label):
        return self.is_support or super().has_module_perms(app_label)

    # not used, needs by django.contrib.auth
    password = models.CharField(
        verbose_name='password',
        max_length=128,
        default='',
    )

    def __str__(self):
        return '%s-%s' % (self.login, self.id)


class Department(mp_tree.MP_Node):
    name_ru = models.CharField(
        max_length=255,
    )
    name_en = models.CharField(
        max_length=255,
        default="",
    )
    slug = models.CharField(
        unique=True,
        max_length=255,
    )

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    @classmethod
    def get_first_root_path(cls):
        return cls._get_path(path=None, depth=1, newstep=1)

    @classmethod
    def get_first_child_path(cls, path):
        depth = cls.get_depth_by_path(path)
        return cls._get_path(path=path, depth=depth + 1, newstep=1)

    @classmethod
    def get_ancestor_paths(cls, path, include_self=False):
        paths = [
            path[:pos]
            for pos in range(
                cls.steplen,
                len(path) + cls.steplen,
                cls.steplen,
            )
        ]
        if not include_self:
            return paths[:-1]
        return paths

    @classmethod
    def get_next_sibling_path(cls, path):
        """
        Видоизмененный _inc_path, который работает с path, а не моделью
        :returns: The path of the next sibling of a given path.
        """
        from treebeard import exceptions

        newpos = cls._str2int(path[-cls.steplen:]) + 1
        key = cls._int2str(newpos)
        if len(key) > cls.steplen:
            msg = "Path Overflow from: '%s'" % path
            raise exceptions.PathOverflow(msg)
        return '{0}{1}{2}'.format(
            path[:-cls.steplen],
            cls.alphabet[0] * (cls.steplen - len(key)),
            key
        )

    @classmethod
    def get_depth_by_path(cls, path):
        return int(len(path) / cls.steplen)

    def __str__(self):
        return '%s-%s' % (self.slug, self.id)


class DepartmentRole(models.Model):
    person = models.ForeignKey(
        to='staff.Person',
        related_name='roles',
        related_query_name='role',
        on_delete=models.CASCADE,
    )
    department = models.ForeignKey(
        to='staff.Department',
        related_name='roles',
        related_query_name='role',
        on_delete=models.CASCADE,
    )
    type = models.CharField(
        max_length=2,
        choices=const.STAFF_ROLE.CHOICES,
        db_index=True,
    )

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta(object):
        unique_together = ('person', 'department', 'type')
        # не уверен, что планировщик запросов будет использовать индекс,
        # но пусть будет, вставка у нас редкая
        index_together = ('person', 'type')

    def __str__(self):
        return '%s %s @ %s' % (
            self.type,
            self.person,
            self.department,
        )


class Subordination(models.Model):
    """
    Денормализация для быстрого получения руководителей и подчиненных
    """
    subject = models.ForeignKey(
        to='staff.Person',
        related_name='subordinates',
        related_query_name='subordinate',
        on_delete=models.CASCADE,
    )
    object = models.ForeignKey(
        to='staff.Person',
        related_name='superiors',
        related_query_name='superior',
        on_delete=models.CASCADE,
    )
    type = models.CharField(
        max_length=1,
        choices=const.SUBORDINATION.CHOICES,
        db_index=True,
    )
    position = models.SmallIntegerField(default=0)
    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta:
        unique_together = ('subject', 'object', 'position')
        index_together = [
            ('object', 'type'),
            ('subject', 'type'),
        ]

    def __str__(self):
        return '%s %s for %s' % (
            self.type,
            self.subject_id,
            self.object_id,
        )


class HR(models.Model):
    cared_person = models.ForeignKey(
        to='staff.Person',
        related_name='hrs',
        related_query_name='hr',
        on_delete=models.CASCADE,
    )
    hr_person = models.ForeignKey(
        to='staff.Person',
        related_name='cared_persons',
        related_query_name='cared_person',
        on_delete=models.CASCADE,
    )
    type = models.CharField(
        max_length=2,
        choices=const.STAFF_ROLE.HR.CHOICES,
    )
    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta:
        indexes = [
            models.Index(['cared_person', 'hr_person']),
            models.Index(['hr_person', 'type']),
        ]


class StaffStructureChange(models.Model):
    date = models.DateField(unique=False)
    staff_id = models.CharField(max_length=255, unique=True)
    processed_at = models.DateTimeField(null=True)

    created_at_auto = models.DateTimeField(auto_now_add=True)
    updated_at_auto = models.DateTimeField(auto_now=True)

    class Meta:
        indexes = [
            models.Index(['date', 'id']),
        ]


class PersonHeads(models.Model):
    structure_change = models.ForeignKey(
        to='staff.StaffStructureChange',
        related_name='person_heads',
        on_delete=models.CASCADE,
    )
    person_id = models.BigIntegerField(null=False)
    heads = models.CharField(
        max_length=255,
        validators=[validators.validate_comma_separated_integer_list],
        null=True
    )

    class Meta:
        indexes = [
            models.Index(['structure_change', 'person_id'])
        ]
