# coding: utf-8
"""
Получение данных из api Стаффа
"""

from requests import Session

from django.conf import settings
from django_ids.services.staff import connector
from django_ids import auth

from review.lib import helpers
from review.lib import std

PERSONS_CHUNK_SIZE = 5000
DEPARTMENTS_CHUNK_SIZE = 5000

staffapi_session = Session()
staffapi_session.headers.update({
    'Authorization': 'OAuth {token}'.format(token=settings.ROBOT_OAUTH_TOKEN),
})


def get_persons_result_set(**params):
    return connector.get_result_set(
        auth=auth.Auth(
            oauth_token=settings.ROBOT_OAUTH_TOKEN,
        ),
        resource='person',
        connector_params={
            'retries': settings.STAFF_API_RETRIES,
            'timeout': settings.STAFF_API_TIMEOUT,
            'host': settings.STAFF_API_HOST,
        },
        params=params,
    )


def get_groups_repo_result_set(**params):
    return connector.get_result_set(
        auth=auth.Auth(
            oauth_token=settings.ROBOT_OAUTH_TOKEN,
        ),
        resource='group',
        connector_params={
            'retries': settings.STAFF_API_RETRIES,
            'timeout': settings.STAFF_API_TIMEOUT,
            'host': settings.STAFF_API_HOST,
        },
        params=params,
    )


@helpers.timeit
def get_roles_result_set():
    url = 'https://{}/v3/departmentstaff'.format(settings.STAFF_API_HOST)

    roles = ('chief', 'deputy', 'hr_partner', 'hr_analyst', 'FINCAB_VIEWER')
    roles_query = ', '.join('"{}"'.format(role) for role in roles)

    fields = ('id', 'role', 'person.login', 'department_group.department.id')

    def make_params(min_id):
        return {
            '_query': 'role in [{}] and id > {}'.format(roles_query, min_id),
            '_fields': ','.join(fields),
            '_limit': 300,
            '_sort': 'id',
            '_nopage': 1,
            'department_group.is_deleted': 'false',
        }

    min_id = -1
    result = []
    while True:
        response = staffapi_session.get(url, timeout=settings.STAFF_API_TIMEOUT, params=make_params(min_id)).json()
        if not response['result']:
            return result
        min_id = response['result'][-1]['id']
        result += response['result']


@helpers.timeit
def get_persons_departments_belonging():
    """
    Получаем маппинг
    {login_сотрудника: id_подразделения}
    """
    fields = ','.join([
        'id',
        'login',
        'department_group.department.id',
    ])
    url = 'https://{}/v3/persons'.format(settings.STAFF_API_HOST)

    def make_params(min_id):
        return {
            '_fields': fields,
            '_sort': 'id',
            '_limit': 500,
            '_nopage': 1,
            'department_group.is_deleted': 'false',
            '_query': 'id > {}'.format(min_id),
        }

    min_id = -1
    result = {}
    while True:
        response = staffapi_session.get(url, timeout=settings.STAFF_API_TIMEOUT, params=make_params(min_id)).json()
        if not response['result']:
            return result
        min_id = response['result'][-1]['id']
        for person in response['result']:
            result[person['login']] = person['department_group']['department']['id']


@helpers.timeit
def get_departments(chunk_size=None):
    chunk_size = chunk_size or DEPARTMENTS_CHUNK_SIZE
    _fields = ','.join([
        'department.id',
        'department.url',
        'parent.department.id',
        'department.name.full',
    ])
    params = {
        'type': 'department',
        '_fields': _fields,
        '_sort': 'id',
        '_limit': chunk_size,
    }
    result_set = get_groups_repo_result_set(**params)
    result = []
    for raw_item in result_set:
        result.append({
            'id': std.safe_itemgetter(raw_item, 'department.id'),
            'slug': std.safe_itemgetter(raw_item, 'department.url'),
            'name_ru': std.safe_itemgetter(raw_item, 'department.name.full.ru'),
            'name_en': std.safe_itemgetter(raw_item, 'department.name.full.en'),
            'parent_id': std.safe_itemgetter(raw_item, 'parent.department.id'),
        })
    return result


def get_persons_from_staff_paged(chunk_size=None):
    """
    Обрабатываем данные чанками, экономим память (
        ориентировочно одна страница 5000 сотрудников ~ 40Мб,
        все вместе ~180Мб
    ), поэтому получаем страницами.
    Если памяти совсем мало, то можно снизить chunk_size, но это удлиннит
    транзакцию.
    """
    chunk_size = chunk_size or PERSONS_CHUNK_SIZE
    _fields = ','.join([
        'uid',
        'login',
        'name.first',
        'name.last',
        'personal.gender',
        'official.is_dismissed',
        'official.position',
        'official.join_at',
        'official.quit_at',
        'official.is_robot',
        'location.office.city.name',
        'language.ui',
        'work_email',
        'environment.timezone',
    ])
    params = {
        '_fields': _fields,
        '_sort': 'id',
        '_limit': chunk_size,
    }
    result_set = get_persons_result_set(**params)
    return result_set.get_pages()
