import logging
import arrow
import itertools

from django.db import transaction
from django.conf import settings

from ids import exceptions

from review.staff import models
from review.staff import const
from review.staff.sync.connector import StaffConnector
from review.staff.sync.fetch import get_persons_result_set
from review.lib import datetimes

log = logging.getLogger(__name__)


def _heads_as_ids_str(chief_logins, empl_login, login_to_id) -> str:
    chiefs = [
        login_to_id[chief]
        for chief in chief_logins
        if chief != empl_login
    ]
    chiefs_without_duplicates = []
    for chief in chiefs:
        if chief not in chiefs_without_duplicates:
            chiefs_without_duplicates.append(chief)
    return ','.join(str(chief) for chief in chiefs_without_duplicates)


def get_persons_at_staff_change():
    staff_change = (
        models.StaffStructureChange.objects
        .filter(processed_at__isnull=True)
        .order_by('id')
    )
    if not staff_change.exists():
        return
    person_with_heads = fetch_persons_with_heads()
    joined_logins_set = joined_today_and_yesterday()
    current_persons_heads_from_db = dict(
        models.PersonHeads.objects
        .order_by('person_id', '-structure_change_id')
        .distinct('person_id')
        .values_list('person_id', 'heads')
    )

    login_to_id = dict(models.Person.objects.values_list('login', 'id'))
    existing = set(login_to_id)
    not_synced = joined_logins_set - existing
    new_persons = []
    for person_with_head in person_with_heads['chiefs']:
        cur_login = person_with_head['person']['login']
        chief_logins = [chief['login'] for chief in person_with_head['chiefs']]

        if not_synced & ({cur_login} | set(chief_logins)):
            # will be synced after models.Person sync
            continue

        person_id = login_to_id[cur_login]
        heads_ids_str = _heads_as_ids_str(chief_logins, cur_login, login_to_id)
        if person_id in current_persons_heads_from_db:
            if heads_ids_str == current_persons_heads_from_db[person_id]:
                # nothing changes in heads
                continue

        last_structure_change = staff_change.last()
        new_persons.append(
            models.PersonHeads(
                structure_change=last_structure_change,
                person_id=person_id,
                heads=heads_ids_str,
            )
        )
    with transaction.atomic():
        models.PersonHeads.objects.bulk_create(new_persons)
        updated = staff_change.update(processed_at=datetimes.now())
    return updated


def fetch_persons_with_heads():
    connector = StaffConnector()

    try:
        response = connector.get(resource=const.PERSON_WITH_HEADS)
    except exceptions.BackendError as exc:
        response_content = exc.response and exc.response.content
        log.exception('Staff bad response with `%s`', response_content)
        return {}

    return response


def get_empty_structure_changes(created_before_time=None):
    if created_before_time is None:
        created_before_time = arrow.now().replace(
            minutes=-settings.NO_PERSON_FOR_STRUCTURE_CHANGE_MINUTES
        ).datetime
    return models.StaffStructureChange.objects.filter(
        person_heads__isnull=True,
        created_at__lte=created_before_time
    )


def joined_today_and_yesterday() -> set:
    today = datetimes.today()
    joined_today = get_persons_result_set(**{
        '_fields': 'login',
        'official.join_at': today.strftime('%Y-%m-%d'),
    })
    yesterday = datetimes.shifted(today, days=-1)
    joined_yesterday = get_persons_result_set(**{
        '_fields': 'login',
        'official.join_at': yesterday.strftime('%Y-%m-%d'),
    })
    logins = set()
    all_joined = itertools.chain(
        joined_today.get_pages(),
        joined_yesterday.get_pages(),
    )
    for page in all_joined:
        for person in page:
            logins.add(person['login'])
    return logins
