# coding: utf-8

import logging
from collections import (
    defaultdict,
    namedtuple,
)

import arrow
import yenv
from django.db import transaction
from django.db.models import OuterRef, Q, Subquery
from ylog import context as log_context

from review.lib import helpers
from review.lib import std
from review.staff import models as staff_models
from review.staff.sync import fetch
from review.staff import const

log = logging.getLogger(__name__)

PERSONS_BULK_UPDATE_SIZE = 1000
PERSONS_MODEL_FIELDS = (
    'id',
    'uid',
    'login',
    'is_dismissed',
    'is_robot',
    'gender',
    'first_name_ru',
    'last_name_ru',
    'first_name_en',
    'last_name_en',
    'city_name_ru',
    'city_name_en',
    'position_ru',
    'position_en',
    'join_at',
    'quit_at',
    'work_email',
    'language',
    'timezone',
)
DEPARTMENTS_MODEL_FIELDS = (
    'id',
    'name_ru',
    'name_en',
    'slug',
)
API_TO_DB_GENDERS_MAP = {
    'male': const.GENDER.MALE,
    'female': const.GENDER.FEMALE,
}
DB_TO_API_FIELDS_MAP = {
    'id': 'id',
    'uid': 'uid',
    'login': 'login',
    'work_email': 'work_email',
    'gender': 'personal.gender',
    'is_dismissed': 'official.is_dismissed',
    'is_robot': 'official.is_robot',
    'first_name_ru': 'name.first.ru',
    'last_name_ru': 'name.last.ru',
    'first_name_en': 'name.first.en',
    'last_name_en': 'name.last.en',
    'city_name_ru': 'location.office.city.name.ru',
    'city_name_en': 'location.office.city.name.en',
    'position_ru': 'official.position.ru',
    'position_en': 'official.position.en',
    'join_at': 'official.join_at',
    'quit_at': 'official.quit_at',
    'language': 'language.ui',
    'timezone': 'environment.timezone',
}


def get_field_from_api_obj(api_obj, db_field):
    api_key = DB_TO_API_FIELDS_MAP[db_field]
    value = std.safe_itemgetter(api_obj, api_key)
    if db_field == 'gender':
        return API_TO_DB_GENDERS_MAP.get(value, const.GENDER.MALE)
    elif db_field == 'uid':
        return int(value)
    elif db_field in ('join_at', 'quit_at') and value is not None:
        arrow_date = arrow.get(value)
        return arrow_date and arrow_date.date()
    else:
        return value


# public
@helpers.timeit
def sync_staff_full():
    stats = {}
    stats.update(sync_staff_structure())
    stats.update(sync_only_person_fields())
    log.info('sync_staff_full stats: %s', stats)
    return stats


@helpers.timeit
def sync_staff_structure():
    """
    Сохранить новых сотрудников, дропнуть все подразделения (
        удаляются DepartmentRole и department_id = NULL для Persons
    ), пересоздать структуру подразделений и проставить сотрудникам
    department_id и создать новые DepartmentRole.
    """
    stats = {}
    # think about new persons between fetches and somehow ignore them
    # that can cause inconsistencies and fails on inserts
    persons_to_department_id = fetch.get_persons_departments_belonging()
    raw_deps_data = fetch.get_departments()
    parents_to_children = build_parents_to_children_mapping(raw_deps_data)
    raw_roles_data = fetch.get_roles_result_set()
    departments_roles_list = build_department_roles_list(raw_roles_data, raw_deps_data)
    department_instances = build_department_instances(parents_to_children)
    # нельзя обновлять подразделения до того, как не получим актуальный состав
    # сотрудников, потому что на них могут быть ссылки в руководителях
    with transaction.atomic():
        stats['created_persons'] = update_person_models()
        person_login_to_id = dict(staff_models.Person.objects.values_list('login', 'id'))
        to_create, to_delete = get_hr_data_diff(
            persons_to_department_id=persons_to_department_id,
            parents_to_children=parents_to_children,
            departments_roles_list=departments_roles_list,
            person_login_to_id=person_login_to_id,
        )
        subordination_data = build_subordination_data(
            persons_to_department_id=persons_to_department_id,
            parents_to_children=parents_to_children,
            departments_roles_list=departments_roles_list,
            person_login_to_id=person_login_to_id,
        )
        # структура подразделений их состав и руководство должны меняться
        # атомарно
        stats['created_departments'] = bulk_create_departments(department_instances)
        stats['person_departments_set'] = update_person_deparments_fast(
            persons_to_department_id,
        )
        stats['created_department_roles'] = bulk_create_department_roles(
            departments_roles_list,
            person_login_to_id,
        )
        stats.update(update_subordination(subordination_data))
        stats.update(update_hr_data(to_create, to_delete))

    return stats


@helpers.timeit
def sync_only_person_fields():
    """
    Вынесено отдельно, потому что это не влияет на структуру.
    Обычно работает недолго, потому что обновляются поля сотрудников редко
    """
    return {'updated_persons': update_existing_persons()}


# about persons
@helpers.timeit
def update_person_models(chunk_size=None):
    created_count = 0
    all_remote_logins = set()
    for page in fetch.get_persons_from_staff_paged(chunk_size=chunk_size):
        remote_persons = get_login_to_person_data(page)
        local_persons_logins = list(
            staff_models.Person.objects
            .filter(login__in=remote_persons)
            .values_list('login', flat=True)
        )
        remote_logins_set = set(remote_persons)
        all_remote_logins |= remote_logins_set
        new_persons_logins = remote_logins_set - set(local_persons_logins)
        created_count += create_persons_bulk(persons_data=[
            person_data
            for person_login, person_data in remote_persons.items()
            if person_login in new_persons_logins
        ])
    need_to_del_nonexisting_persons = yenv.choose_key_by_type(
        {
            'testing': True,
            'production': False,
        },
        fallback=True,
    )
    if need_to_del_nonexisting_persons:
        # Not real users can appear on staff.test
        # And theay desappear after db sync from staff.prod
        # So we need to del them only on staff.test
        deleted = _del_nonexisting_persons(all_remote_logins)
        log.info('Deleted persons result %s', deleted)
    return created_count


@helpers.timeit
def update_existing_persons(chunk_size=None):
    updated_count = 0
    for page in fetch.get_persons_from_staff_paged(chunk_size=chunk_size):
        remote_persons = get_login_to_person_data(page)
        local_persons = get_person_data_from_db(logins=list(remote_persons.keys()))

        updated_count += update_persons_if_changed(
            old_persons_data=local_persons,
            new_persons_data={
                person_login: person_data
                for person_login, person_data in remote_persons.items()
                if person_login in local_persons
            },
        )
    return updated_count


def get_login_to_person_data(page):
    if yenv.type not in ('unstable', 'testing'):
        login_to_person_data = {person['login']: person for person in page}
    else:
        login_to_person_data = {person['login']: person for person in page if 'login' in person}
    return login_to_person_data


@helpers.timeit
def get_existing_person_ids(ids):
    return list(staff_models.Person.objects.filter(
        id__in=ids).values_list('id', flat=True))


@helpers.timeit
def get_person_data_from_db(logins):
    return {
        obj['login']: obj
        for obj in staff_models.Person.objects.filter(
            login__in=logins
        ).values(*PERSONS_MODEL_FIELDS)
    }


@helpers.timeit
def create_persons_bulk(persons_data):
    Person = staff_models.Person

    new_persons = []
    for person_data in persons_data:
        model_data = {
            db_key: get_field_from_api_obj(person_data, db_key)
            for db_key in DB_TO_API_FIELDS_MAP
        }
        new_persons.append(Person(**model_data))
    return len(Person.objects.bulk_create(new_persons, batch_size=500))


@helpers.timeit
def update_persons_if_changed(old_persons_data, new_persons_data):
    updated_count = 0
    for login, new_person_data in new_persons_data.items():
        old_person_data = old_persons_data[login]
        with log_context.LogContext(instance='Person:{}'.format(login)):
            is_updated = update_one_person_if_changed(
                old_person_data,
                new_person_data,
            )
        if is_updated:
            updated_count += 1
    return updated_count


def update_one_person_if_changed(old_person_data, new_person_data):
    diff = get_person_data_diff(old_person_data, new_person_data)
    if not diff:
        return False

    fields_for_update = {}
    for db_field, old_val, new_val in diff:
        fields_for_update[db_field] = new_val

    staff_models.Person.objects.filter(
        login=old_person_data['login']
    ).update(**fields_for_update)
    return True


def get_person_data_diff(old_person_data, new_person_data):
    diff = []
    for db_field in set(DB_TO_API_FIELDS_MAP) - {'id'}:
        old_value = old_person_data[db_field]
        new_value = get_field_from_api_obj(
            api_obj=new_person_data,
            db_field=db_field,
        )
        if old_value != new_value:
            diff.append((db_field, old_value, new_value))

    if diff:
        log.debug('Person id=%d needs update %s', old_person_data['id'], diff)

    return diff


@helpers.timeit
def update_person_deparments(persons_to_department_id):
    for person in staff_models.Person.objects.only('id'):
        person.department_id = persons_to_department_id.get(person.id)
        person.save(force_update=True)


@helpers.timeit
def update_person_deparments_fast(persons_to_department_id):
    from django_bulk_update.helper import bulk_update

    all_persons = staff_models.Person.objects.only('login')
    for person in all_persons:
        person.department_id = persons_to_department_id.get(person.login)
    return bulk_update(
        objs=all_persons,
        update_fields=['department_id'],
        batch_size=PERSONS_BULK_UPDATE_SIZE,
    )


# about departments and department roles
@helpers.timeit
def bulk_create_departments(instances):
    staff_models.Department.objects.all().delete()
    created_count = len(
        staff_models.Department.objects.bulk_create(
            objs=instances,
            batch_size=500,
        )
    )
    return created_count


@helpers.timeit
def bulk_create_department_roles(department_roles_list, person_login_to_id):
    DepRole = namedtuple('DepRole', 'department_id, person_id, type')
    roles = {
        DepRole(item['department_id'], person_login_to_id[item['person_login']], item['type'])
        for item in department_roles_list
    }
    objects = [
        staff_models.DepartmentRole(id=id_, **role._asdict())
        for id_, role in enumerate(roles, start=1)
    ]
    return len(staff_models.DepartmentRole.objects.bulk_create(objs=objects))


@helpers.timeit
def build_subordination_data(
    persons_to_department_id,
    parents_to_children,
    departments_roles_list,
    person_login_to_id,
):
    old_data = fetch_subordination_from_db()
    new_data = fetch_subordination_from_api_data(
        persons_to_department_id,
        parents_to_children,
        departments_roles_list,
        person_login_to_id,
    )
    new_data |= fetch_subordination_dismissed()
    return {
        'models_to_create': build_subordination_models(
            data_set=new_data - set(old_data)
        ),
        'ids_to_delete': [old_data[datum] for datum in set(old_data) - new_data],
    }


@helpers.timeit
def fetch_subordination_from_db():
    queryset = staff_models.Subordination.objects.values_list(
        'subject_id',
        'object_id',
        'type',
        'position',
        'id',
    )
    return {
        (subject_id, object_id, type, position): id
        for subject_id, object_id, type, position, id in queryset
    }


def fetch_subordination_dismissed():
    data = (
        staff_models.Person.objects
        .filter(is_dismissed=True)
        .annotate(chief_chain=Subquery(
            staff_models.StaffStructureChange.objects
            .filter(date__lt=OuterRef('quit_at'))
            .filter(person_heads__person_id=OuterRef('pk'))
            .order_by('-date', '-created_at_auto')
            .distinct('date')
            .values('person_heads__heads')[:1]
        ))
        .filter(chief_chain__isnull=False)
        .exclude(chief_chain='')
        .values_list('id', 'chief_chain')
    )
    res = set()
    for person_id, chief_chain in data:
        for pos, chief_id in enumerate(chief_chain.split(',')):
            type_ = const.SUBORDINATION.DIRECT if not pos else const.SUBORDINATION.INDIRECT
            res.add((
                chief_id,
                person_id,
                type_,
                pos,
            ))
    return res


@helpers.timeit
def fetch_subordination_from_api_data(
    persons_to_department_id,
    parents_to_children,
    departments_roles_list,
    person_login_to_id,
):
    result = set()

    not_dismissed_persons = set(staff_models.Person.objects.filter(
        is_dismissed=False,
    ).values_list('login', flat=True))
    dep_to_head = {
        role['department_id']: role['person_login']
        for role in departments_roles_list
        if role['type'] == const.STAFF_ROLE.DEPARTMENT.HEAD
    }
    child_dep_to_parent = {}
    for parent_id, children in parents_to_children.items():
        for child in children:
            child_dep_to_parent[child['id']] = parent_id

    for person_login, person_department_id in persons_to_department_id.items():
        if person_login not in not_dismissed_persons:
            # тех, кто уже уволен нам не нужно добавлять в Subordination
            continue
        departments_chain = []
        superiors_chain = []

        next_department_id = person_department_id
        while next_department_id:
            departments_chain.append(next_department_id)
            next_department_id = child_dep_to_parent.get(next_department_id)
        for dep_id in departments_chain:
            head = dep_to_head.get(dep_id)
            if head is not None and head != person_login:
                superiors_chain.append(person_login_to_id[head])

        std.remove_adjacent_doubles(superiors_chain)
        for position, superior_id in enumerate(superiors_chain):
            if position == 0:
                type = const.SUBORDINATION.DIRECT
            else:
                type = const.SUBORDINATION.INDIRECT
            result.add((
                superior_id,
                person_login_to_id[person_login],
                type,
                position,
            ))
    return result


@helpers.timeit
def build_subordination_models(data_set):
    models = []
    for datum in data_set:
        subject_id, object_id, type, position = datum
        models.append(
            staff_models.Subordination(
                subject_id=subject_id,
                object_id=object_id,
                type=type,
                position=position,
            )
        )
    return models


@helpers.timeit
def update_subordination(subordination_data):
    delete_subordination(id__in=subordination_data['ids_to_delete'])
    created = staff_models.Subordination.objects.bulk_create(
        objs=subordination_data['models_to_create'],
        batch_size=500,
    )
    return {'created_subordination': len(created)}


@helpers.timeit
def delete_subordination(**filter_params):
    staff_models.Subordination.objects.filter(**filter_params).delete()


@helpers.timeit
def update_hr_data(to_create, to_delete):
    staff_models.HR.objects.filter(id__in=to_delete).delete()
    created = staff_models.HR.objects.bulk_create(
        objs=to_create,
        batch_size=500,
    )
    return dict(
        created_hrs=len(created),
        deleted_hrs=len(to_delete),
    )


HRData = namedtuple('HR', ('cared_id', 'hr_id', 'type'))


def get_hr_data_diff(
    persons_to_department_id,
    parents_to_children,
    departments_roles_list,
    person_login_to_id,
):
    db_hr_data_to_id = fetch_hrs_from_db()
    api_hr_data = fetch_hrs_from_api_data(
        persons_to_department_id,
        parents_to_children,
        departments_roles_list,
        person_login_to_id,
    )
    existing_db_hrs = set(db_hr_data_to_id)
    to_create = [
        staff_models.HR(
            cared_person_id=hr_data.cared_id,
            hr_person_id=hr_data.hr_id,
            type=hr_data.type,
        )
        for hr_data in api_hr_data - existing_db_hrs
    ]
    to_delete = [db_hr_data_to_id[key] for key in existing_db_hrs - api_hr_data]
    return to_create, to_delete


@helpers.timeit
def fetch_hrs_from_db():
    hrs = staff_models.HR.objects.values_list(
        'cared_person_id',
        'hr_person_id',
        'type',
        'id',
    )
    return {
        HRData(cared_id, hr_id, hr_type): id
        for cared_id, hr_id, hr_type, id in hrs
    }


@helpers.timeit
def fetch_hrs_from_api_data(
    persons_to_department_id,
    parents_to_children,
    departments_roles_list,
    person_login_to_id,
):
    dep_to_hrs = defaultdict(list)
    hrs = (
        (r['department_id'], r['person_login'], r['type'])
        for r in departments_roles_list
        if r['type'] in const.STAFF_ROLE.HR.ALL
    )
    for dep_id, hr_login, hr_type in hrs:
        dep_to_hrs[dep_id].append((person_login_to_id[hr_login], hr_type))

    child_dep_to_parent = {}
    for parent_id, children in parents_to_children.items():
        child_dep_to_parent.update((child['id'], parent_id) for child in children)

    res = set()
    for person_login, dep_id in persons_to_department_id.items():
        person_hrs = []
        while dep_id is not None:
            person_hrs += dep_to_hrs.get(dep_id, [])
            dep_id = child_dep_to_parent.get(dep_id)
        res |= {
            HRData(person_login_to_id[person_login], hr_id, hr_type)
            for hr_id, hr_type in person_hrs
            if hr_id != person_login_to_id[person_login]
        }
    return res


def build_parents_to_children_mapping(data):
    mapping = {}
    for node in data:
        parent_id = node.get('parent_id')
        children = mapping.setdefault(parent_id, [])
        children.append({
            key: node[key]
            for key in DEPARTMENTS_MODEL_FIELDS
        })
    return mapping


def build_department_roles_list(raw_roles_data, raw_deps_data):
    result = []
    staff_api_to_review = {
        'chief': const.STAFF_ROLE.DEPARTMENT.HEAD,
        'deputy': const.STAFF_ROLE.DEPARTMENT.DEPUTY,
        'hr_partner': const.STAFF_ROLE.HR.HR_PARTNER,
        'hr_analyst': const.STAFF_ROLE.HR.HR_ANALYST,
        'FINCAB_VIEWER': const.STAFF_ROLE.HR.FINANCE_VIEWER,
    }
    active_deps_ids = {dep['id'] for dep in raw_deps_data}

    for role_record in raw_roles_data:
        role_type = staff_api_to_review.get(role_record['role'])
        role_department_id = role_record['department_group']['department']['id']

        if role_type is not None and role_department_id in active_deps_ids:
            result.append({
                'department_id': role_department_id,
                'person_login': role_record['person']['login'],
                'type': role_type,
            })

    return result


def build_department_instances(parents_to_children):
    from queue import Queue

    model_cls = staff_models.Department

    queue, result = Queue(), []

    node_id = node = left_sibling = None
    children = parents_to_children.get(node_id, [])
    for child in children:
        child = dict(child)
        queue.put((child, node, left_sibling))
        left_sibling = child

    while not queue.empty():
        node, parent, left_sibling = queue.get()

        if left_sibling is None:
            if parent is None:
                path = model_cls.get_first_root_path()
            else:
                path = model_cls.get_first_child_path(parent['path'])
        else:
            path = model_cls.get_next_sibling_path(
                path=left_sibling['path'])

        children = parents_to_children.get(node['id'], [])
        node.update(
            path=path,
            depth=model_cls.get_depth_by_path(path),
            numchild=len(children),
        )
        result.append(model_cls(**node))

        left_sibling = None
        for child in children:
            child = dict(child)
            queue.put((child, node, left_sibling))
            left_sibling = child

    return result


def fix_incorrect_sync():
    """
    Функция фикса некорректного синка со стаффом (только для тестинга)
    Удаляет пользователей, если со стаффа приходит новый пользователь с таким же uid-ом
    """
    if yenv.type in ('production', 'prestable'):
        raise RuntimeError('Only for non-prod database')
    remote_persons = {}
    for page in fetch.get_persons_from_staff_paged(chunk_size=None):
        remote_persons.update((person['uid'], person['login']) for person in page)
    local_persons = dict(staff_models.Person.objects.values_list('uid', 'login'))
    count = 0
    for uid, login in local_persons.items():
        count += 1
        if remote_persons.get(str(uid)) not in (None, login):
            staff_models.Person.objects.filter(uid=uid).delete()
            print('Deleted {}'.format(login))
        if not (count % 100):
            print('Processed {}'.format(count))
    sync_staff_full()


def _del_nonexisting_persons(remote_logins):
    db_logins = set(
        staff_models.Person.objects
        .values_list('login', flat=True)
    )
    to_del = db_logins - remote_logins
    return (
        staff_models.Person.objects
        .filter(login__in=to_del).delete()
    )
