# coding: utf-8

import logging

import arrow
from django.views.decorators import csrf
from django.utils.decorators import method_decorator
from django.db import IntegrityError
from django.conf import settings
from django import forms

from review.lib import views
from review.lib import errors
from review.staff import models
from review.staff import tasks


log = logging.getLogger(__name__)


class StaffChangeForm(forms.Form):
    proposal_id = forms.CharField()


@method_decorator(csrf.csrf_exempt, name='dispatch')
class StaffChangeView(views.View):
    form_cls_post = StaffChangeForm

    def has_access(self, auth):
        app_id = getattr(auth, 'tvm_app_id', None)
        return (
            auth.login == settings.ROBOT_STAFF_LOGIN or
            (app_id is not None and app_id == settings.TVM_APP_IDS['staff'])
        )

    def process_post(self, auth, data):
        proposal_id = data['proposal_id']

        if not self.has_access(auth):
            raise errors.PermissionDenied(
                type='staff_change',
                allowed_logins=[settings.ROBOT_STAFF_LOGIN],
                proposal_id=proposal_id,
            )

        try:
            models.StaffStructureChange.objects.create(
                date=arrow.now().date(),
                staff_id=proposal_id,
            )
        except IntegrityError as exc:
            # it is ok to receive constraint error
            err_msg = exc.args[0].lower()
            if 'unique constraint' not in err_msg:
                raise

            msg = 'proposal {} exists'.format(proposal_id)
        else:
            msg = 'ok for structure change {}'.format(proposal_id)

        log.info(msg)
        return {'result': msg}
