# coding: utf-8


from datetime import timedelta
import json
import logging
from base64 import b64decode
from collections import Counter

from celery import states
from django.conf import settings
from django.utils import timezone
from django_celery_monitoring.utils import get_muted_task_names

from django_celery_results.models import TaskResult
from review.staff import stats as staff_stats
from review.notifications import stats as notification_stats
from review.lib import views, errors
from review.stats.celery_stats import CeleryStats

log = logging.getLogger(__name__)


class StatView(views.View):

    def process_get(self, auth, data):
        celery_stats = CeleryStats()
        return (
            staff_stats.get_stat() +
            notification_stats.get_stat() +
            celery_stats.get_stat()
        )


class CeleryMonitoringView(views.View):
    include_auth_to_response = False

    def process_get(self, auth, data):
        return self.get_celery_stat()

    @classmethod
    def get_celery_stat(cls):
        def get_task_name(task):
            payload = json.loads(task['payload'])
            body = json.loads(b64decode(payload['body']))
            if isinstance(body, dict):
                return body.get('task')
            headers = payload.get('headers')
            if isinstance(headers, dict):
                return headers.get('task')
            return ''

        celery_stats = CeleryStats()
        queue_size = celery_stats.get_queue_size()

        if queue_size:
            log.info('Celery queue size: %s', queue_size)

        if queue_size > settings.MAX_CELERY_QUEUE_SIZE:
            counter = Counter(
                map(
                    get_task_name,
                    celery_stats.messages.find(celery_stats.query).limit(5000)
                )
            )

            return {
                'message': 'Too many tasks in Celery queue: {queue_size} (more than {limit})'.format(
                    queue_size=queue_size,
                    limit=settings.MAX_CELERY_QUEUE_SIZE,
                ),
                'queue_size': queue_size,
                'tasks': counter
            }

        return {}


class CeleryFailedTasksView(views.View):
    include_auth_to_response = False

    def process_get(self, auth, data):
        """
        ручка вернет упавшие таски за последнюю неделю, так как более давние
        автоматически удаляются таской review.core.tasks.clean_old_celery_results_task
        """
        task_names = list(
            TaskResult.objects
            .filter(status=states.FAILURE)
            .exclude(task_name__in=get_muted_task_names())
            .order_by('task_name')
            .values_list('task_name', flat=True)
            .distinct()
        )

        return {'task_names': task_names}


class DbIsAvailableView(views.View):

    def process_get(self, auth, data):
        try:
            # random query
            staff_stats.models.StaffStructureChange.objects.first()
            return {}
        except:
            raise errors.Error('db_is_not_ready')
