import logging
from django.conf import settings

from review.celery_app import app
from review.shortcuts import const
from review.shortcuts import models
from review.core.logic import assemble
from review.xiva.tasks import send_post_to_xiva


FIELDS = const.FIELDS
XIVA_SEND_CPR_URL_FORMAT = settings.XIVA_SEND_URL_FORMAT.format(
    topic='calibration{id}',
    event='update_person_review'
)

log = logging.getLogger(__name__)


def get_message_for_xiva(calibration_id, calibration_person_reviews):
    person_review_list = []
    for cpr in calibration_person_reviews:
        if cpr.calibration_id == calibration_id:
            person_review_list.append({
                'login': cpr.person_review.person_login,
                'person_review_id': cpr.person_review.id,
            })
    return {'list_pr': person_review_list}


def post_to_xiva(login, person_review_ids):
    if len(person_review_ids) == 1:
        post_new_action_to_xiva(login, person_review_ids, delay=False)
    elif person_review_ids:
        post_new_action_to_xiva.delay(login, person_review_ids)


@app.task
def post_new_action_to_xiva(login, person_review_ids, delay=True):
    user = models.Person.objects.get(login=login)

    calibration_person_reviews = assemble.get_calibration_person_reviews(
        subject=user,
        filters_chosen={
            'person_review_ids': person_review_ids,
            'calibration_status': [const.CALIBRATION_STATUS.IN_PROGRESS],
        },
        requested_person_review_fields={
            FIELDS.ID,
            FIELDS.PERSON_LOGIN,
        }
    )
    ids_post = {cpr.calibration_id for cpr in calibration_person_reviews}

    for calibration_id in ids_post:
        url = XIVA_SEND_CPR_URL_FORMAT.format(id=calibration_id)
        message = get_message_for_xiva(calibration_id, calibration_person_reviews)
        send_post_to_xiva(url=url, message=message, delay=delay)
