import time
from contextlib import contextmanager
from copy import deepcopy
from intranet.search.core.utils import QueryDict


class Timer:
    def __init__(self):
        self.duration = 0

    @contextmanager
    def __call__(self):
        start = time.time()

        yield

        self.duration = time.time() - start


class TriggerContext(dict):
    @classmethod
    def initial(cls, trigger, **kwargs):
        defaults = {
            'trigger': trigger,
            'weight': 0,
            'triggered': False,

            'triggers': [],
            'weights': [],
            'attributes': QueryDict(),
        }

        return cls(defaults, **kwargs)

    @contextmanager
    def __call__(self, trigger):
        trigger_context = self.__class__.initial(
            trigger=trigger, qtree=self['qtree'],
            state=self['state'], wizard=self['wizard'])

        timer = Timer()

        with timer():
            yield trigger_context

        trigger_context['duration'] = timer.duration

        self['qtree'] = trigger_context['qtree']
        self['attributes'] = trigger_context.get('attributes')
        self['weights'].append(trigger_context['weight'] + sum(trigger_context['weights']))

        self['triggers'].append(trigger_context)

    def prepare(self):
        triggers = self.pop('triggers')
        self['triggers'] = [t.prepare() for t in triggers]
        self['trigger'] = str(self['trigger'])
        return dict(self)


class WizardContext(dict):
    @classmethod
    def initial(cls, wizard, **kwargs):
        defaults = {
            'wizard': wizard,
            'weights': [],
            'triggers': [],
            'triggered': False,
            'attributes': QueryDict(),
        }

        return cls(defaults, **kwargs)

    @contextmanager
    def __call__(self, trigger):
        # TODO: вот тут вероятно и нужно собирать параметры запроса
        # и менять его уже в SearchTrigger если так заявлено в этом триггере (и то только qtree)

        trigger_context = TriggerContext.initial(
            trigger=trigger, qtree=self['qtree'],
            state=self['state'], wizard=self['wizard'])

        timer = Timer()

        with timer():
            yield trigger_context

        trigger_context['duration'] = timer.duration

        self['qtree'] = trigger_context['qtree']
        self['attributes'].update(trigger_context['attributes'])
        self['weights'].append(trigger_context['weight'] + sum(trigger_context['weights']))

        self['triggers'].append(trigger_context)

    def prepare(self):
        triggers = self.pop('triggers')
        self['triggers'] = [t.prepare() for t in triggers]
        self['wizard'] = str(self['wizard'])
        return dict(self)


class SelectContext(dict):
    @classmethod
    def initial(cls, state, **kwargs):
        defaults = {
            'qtree': state.qtree.clone(),
            'wizards': [],
            'weights': [],
            'state': state,
        }

        return cls(defaults, **kwargs)

    def copy(self):
        return deepcopy(self)

    @contextmanager
    def __call__(self, wizard):
        wizard_context = WizardContext.initial(
            wizard=wizard, qtree=self['qtree'],
            state=self['state'])

        timer = Timer()

        with timer():
            yield wizard_context

        wizard_context['duration'] = timer.duration

        self['qtree'] = wizard_context['qtree']
        self['attributes'] = wizard_context['attributes']
        self['weights'].extend(wizard_context['weights'])
        self['wizards'].append(wizard_context)

    def prepare(self):
        wizards = self.pop('wizards')
        self['wizards'] = [w.prepare() for w in wizards]
        return dict(self)
