import json
import logging
import re

from tornado import gen
from django.conf import settings

from intranet.search.core.query import parse_query
from intranet.search.core.query.ast import Or, Text
from intranet.search.core.utils.lemmer import get_lemmas

from intranet.search.abovemeta.decider.ranking import get_relevance
from intranet.search.abovemeta.search_settings import prepare_search

logger = logging.getLogger(__name__)
timeouts = settings.ISEARCH['abovemeta']['timeouts']


class Trigger:
    label = 'Триггер'
    weight = 0.

    def __init__(self, **kwargs):
        self.feature_enabled = kwargs.get('feature_enabled')
        self.feature_disabled = kwargs.get('feature_disabled')

    def __str__(self):
        return self.label

    @gen.coroutine
    def __call__(self, context, *args, **kwargs):
        triggered = yield self.check_features(context)
        if not triggered:
            raise gen.Return(False)

        triggered = yield self.check_trigger(context, *args, **kwargs)
        raise gen.Return(triggered)

    @gen.coroutine
    def check_features(self, context):
        if self.feature_enabled and not context['state'].feature_enabled(self.feature_enabled):
            raise gen.Return(False)
        elif self.feature_disabled and context['state'].feature_enabled(self.feature_disabled):
            raise gen.Return(False)
        else:
            raise gen.Return(True)

    @gen.coroutine
    def check_trigger(self, context, *args, **kwargs):
        raise gen.Return(True)


class Always(Trigger):
    label = 'Всегда'

    def __init__(self, weight=0., **kwargs):
        self.weight = weight
        super().__init__(**kwargs)

    @gen.coroutine
    def check_trigger(self, context, *args, **kwargs):
        context['weight'] = self.weight
        raise gen.Return(True)


class All(Trigger):
    label = 'Все'

    def __init__(self, *triggers, **kwargs):
        self.triggers = triggers
        super().__init__(**kwargs)

    @gen.coroutine
    def check_trigger(self, context, *args, **kwargs):
        all_triggers_passed = True
        for trigger in self.triggers:
            with context(trigger) as trigger_context:
                triggered = yield trigger(trigger_context, *args, **kwargs)

                if triggered:
                    trigger_context['triggered'] = True  # изначально стоит False
                else:
                    all_triggers_passed = False

        if not all_triggers_passed:
            context['weights'] = [0] * len(self.triggers)  # веса в контексте добавлялись, когда вызывали подтригеры
            # all упал, но веса остались - убиваем их и получаем занулённый all
            raise gen.Return(False)
        raise gen.Return(True)


class Any(Trigger):
    label = 'Любой'

    def __init__(self, *triggers, **kwargs):
        self.triggers = triggers
        super().__init__(**kwargs)

    @gen.coroutine
    def check_trigger(self, context, *args, **kwargs):
        done = False

        for trigger in self.triggers:
            with context(trigger) as trigger_context:
                triggered = yield trigger(trigger_context, *args, **kwargs)
                if triggered:
                    trigger_context['triggered'] = True
                    done = True

        raise gen.Return(done)


class Optional(Trigger):
    label = 'Опциональный'

    def __init__(self, trigger, **kwargs):
        self.trigger = trigger
        super().__init__(**kwargs)

    @gen.coroutine
    def check_trigger(self, context, *args, **kwargs):
        with context(self.trigger) as trigger_context:
            triggered = yield self.trigger(trigger_context, *args, **kwargs)

            if triggered:
                trigger_context['triggered'] = True

        raise gen.Return(True)


class ContentTrigger(Trigger):
    label = 'Шаблон'

    def __init__(self, pattern, weight=0.0, replace_query=True, **kwargs):
        if isinstance(pattern, str):
            pattern = [pattern]
        self.pattern = get_lemmas(pattern)
        self.weight = weight
        self.replace_query = replace_query
        super().__init__(**kwargs)

    def __str__(self):
        return '{}: {}'.format(self.label, ', '.join(self.pattern))

    @gen.coroutine
    def check_trigger(self, context, *args, **kwargs):
        qtree = context['qtree'].clone() if self.replace_query else context['qtree']
        triggered = False

        for node in qtree.filter_text_nodes():
            lemmas = get_lemmas([node.text])
            if lemmas & self.pattern:
                triggered = True
                context['weight'] = self.weight
                if self.replace_query:
                    node.text = ''

        if self.replace_query and qtree.to_string().strip() != '':
            context['qtree'] = qtree

        raise gen.Return(triggered)


class RegexTrigger(Trigger):
    label = 'Регулярка'

    def __init__(self, pattern, replace=None, replace_with_query=False, weight=0.0, **kwargs):
        self.pattern = pattern
        self.replace = replace
        self.weight = weight
        self.replace_with_query = replace_with_query
        super().__init__(**kwargs)

    def __str__(self):
        return f'{self.label}: {self.pattern}'

    @gen.coroutine
    def check_trigger(self, context, *args, **kwargs):
        if self.replace:
            qtree = context['qtree'].clone()
        else:
            qtree = context['qtree']

        text_nodes = list(qtree.filter_text_nodes())
        triggered = False

        for node in text_nodes:
            if re.search(self.pattern, node.text):
                triggered = True
                if self.replace is not None:
                    if self.replace_with_query:
                        qtree = parse_query(
                            re.sub(self.pattern, self.replace, node.text))
                        break
                    else:
                        node.text = re.sub(self.pattern, self.replace, node.text)

        if triggered:
            context['weight'] = self.weight

        context['qtree'] = qtree
        raise gen.Return(triggered)


class SearchTrigger(Trigger):
    label = 'Запрос'

    def __init__(self, search, collection=None, weight=0.5,
                 query=parse_query('#{qtree}#'),
                 replace_query=False, split_text=False, **kwargs):
        super().__init__(**kwargs)
        self.search = search
        self.collection = collection or ''
        self.weight = weight
        self.query = query
        self.replace_query = replace_query
        self.split_text = split_text

    def __str__(self):
        res = self.query.format(qtree=Text('{qtree}'), original=Text('{original}')).to_string()
        return f'{self.label}: {res}'

    @gen.coroutine
    def check_trigger(self, context, requester=None, *args, **kwargs):
        if context['qtree'].to_string() == '':
            raise gen.Return(False)

        if self.replace_query:
            qtree = context['qtree'].clone()
        else:
            qtree = context['qtree']

        if self.split_text:
            qtree = Or.join(qtree.filter_text_nodes())

        search_settings = prepare_search(context['state'], self.search, self.collection)
        if not search_settings:
            logger.error('Cannot get search settings. search %s, index: %s, organization_id: %s',
                         self.search, self.collection, context['state'].org_directory_id)
            raise gen.Return(False)

        # FIXME подумать, как правильно организовать код, чтобы не использовать стрёмные словари
        search = {
            'name': context['wizard'].name,
            'search_settings': search_settings,
            'qtree': qtree,
            'template': self.query,
        }
        if search_settings['wizard_rules']:
            search['wizard_rules'] = search_settings['wizard_rules']

        state = context['state']
        request_builder = search_settings['request_builder'](search, state)
        request_builder.apply_haha()
        request = request_builder.get_request(
            type='decider',
            request_timeout=timeouts['decider'],
            retries=1,
        )
        response = yield requester.fetch(request)

        if response is None:
            raise gen.Return(False)

        try:
            parsed = search_settings['response_class'](json.load(response.buffer))
            is_on = parsed.count
        except Exception as e:
            logger.exception(e)
            raise gen.Return(False)

        if is_on:
            if state.feature_enabled('ml_wizard_ranking'):
                context['weight'] = get_relevance(state, parsed, context['wizard'].name)
            else:
                context['weight'] = self.weight

        if self.replace_query and is_on:
            context['qtree'] = qtree

        raise gen.Return(is_on)
