import re
from urllib.parse import unquote

from collections import defaultdict
from wtforms import StringField, IntegerField, BooleanField, Form
from wtforms.form import BaseForm
from wtforms.validators import ValidationError

from django.conf import settings


from intranet.search.core.fields import MaskField, SplitField, MultiValueMaskField, SearchQueryField
from intranet.search.core.utils import QueryDict

from intranet.search.core.query import parse_query


ATTR_LOOKUP_LT = 'lt'
ATTR_LOOKUP_LTE = 'lte'
ATTR_LOOKUP_GT = 'gt'
ATTR_LOOKUP_GTE = 'gte'
ATTR_LOOKUP_EQ = 'eq'

ATTR_LOOKUP_TYPES = (
    ATTR_LOOKUP_LT,
    ATTR_LOOKUP_LTE,
    ATTR_LOOKUP_GT,
    ATTR_LOOKUP_GTE,
    ATTR_LOOKUP_EQ,
)


def layer_validator(form, field):
    qd = QueryDict()
    qd.update(field.data)
    f = LayerForm(qd)
    if not f.validate():
        raise ValidationError(f.errors)


class LayerForm(Form):
    order = IntegerField('order')
    query = StringField('query')
    page = IntegerField('page')
    per_page = IntegerField('per_page')

    def validate_query(self, field):
        if not field.data:
            return

        try:
            parse_query(field.data, 5, text_fallback=False)
        except Exception as e:
            raise ValidationError('Could not parse query, error %s' % e)


def layers_validator(form, field):
    av_layers = settings.ISEARCH['suggest']['layers'].keys()
    add_layers = set(field.data or []) - set(av_layers)
    if add_layers:
        raise ValidationError('Unknown layers: %s' % ', '.join(add_layers))


def source_validator(form, field):
    if field.data and field.data not in settings.ISEARCH['suggest']['sources'].keys():
        raise ValidationError('Unknown source: %s' % field.data)


def text_validator(form, field):
    """ Проверяет поле текст в зависимости от наличия флага allow_empty
    """
    field.data = field.data.strip()
    if not field.data and not form.data.get('allow_empty'):
        raise ValidationError('This field is required.')


def callback_validator(form, field):
    """ Валидирует колбэк в jsonp запросе
    """
    if not field.data:
        return

    if not re.match(r'^[a-zA-Z0-9_.]{1,64}$', field.data):
        raise ValidationError('Invalid callback format. '
                              'Allowed symbols: a-zA-Z0-9_., max length: 64')


class AttrLookupFiled(MaskField):
    """ Поле для передачи фильтров вида attr.interviews_with_nohire_count__gte=1
    """
    lookup_separator = '__'

    def process(self, formdata, data=None):
        self.process_errors = []
        data = defaultdict(dict)
        mask = self.mask + self.separator
        for key in formdata:
            if key.startswith(mask):
                lookup = unquote(key[len(mask):]).split(self.lookup_separator, 2)
                if len(lookup) == 1:
                    field_name = lookup[0]
                    lookup_type = ATTR_LOOKUP_EQ
                else:
                    field_name, lookup_type = lookup

                if lookup_type in ATTR_LOOKUP_TYPES:
                    data[lookup_type][field_name] = self.process_value(formdata, key)
                else:
                    self.process_errors.append(
                        'Unknown lookup type "%s". Available types: %s' %
                        (lookup_type, ', '.join(ATTR_LOOKUP_TYPES)))

        self.data = dict(data)


class SearchForm(Form):
    """ Форма обработки параметров поиска
    """
    text = SearchQueryField('text', default='')
    allow_empty = BooleanField('allow_empty')
    language = StringField('language')
    features = MaskField('feature')
    scope = StringField('scope', default='search')
    content = StringField('content')
    backend = StringField('backend')
    facet = MultiValueMaskField('facet')
    zone = MaskField('zone')
    attr = AttrLookupFiled('attr')
    wizards = MaskField('wizard')
    sorted = StringField('sorted')
    sorted_order = StringField('sorted_order')
    p = IntegerField('p')
    debug = BooleanField('debug')
    relev_info = BooleanField('relev_info')
    request_id = StringField('request_id')
    org_id = IntegerField('org_id')
    nomisspell = BooleanField('nomisspell')
    test_id = StringField('test_id')


def version_validator(form, field):
    allowed_versions = {0, 1, 2}
    if field.data not in allowed_versions:
        raise ValidationError('Unknown version: %s. Allowed versions: %s',
                              (field.data, ', '.join(str(a) for a in allowed_versions)))


class SuggestForm(BaseForm):
    def __init__(self, form_data):
        fields = {
            'text': SearchQueryField('text', default='', validators=[text_validator]),
            'language': StringField('language'),
            'features': MaskField('feature'),
            'allow_empty': BooleanField('allow_empty'),
            'source': StringField('source', validators=[source_validator]),
            'layers': SplitField('layers', validators=[layers_validator]),
            's[]': SplitField('layers', validators=[layers_validator]),  # LEGACY, синоним layers
            's': SplitField('layers', validators=[layers_validator]),  # LEGACY, синоним layers
            'callback': StringField('callback', validators=[callback_validator]),
            'version': IntegerField(default=0, validators=[version_validator]),
            'request_id': StringField('request_id'),
            'org_id': IntegerField('org_id'),
            'debug': BooleanField('debug'),
        }

        layers = settings.ISEARCH['suggest']['layers'].keys()

        for layer in layers:
            fields[layer] = MaskField(layer, validators=[layer_validator])

        super().__init__(fields)

        self.process(form_data)
