from collections import defaultdict
from functools import partial, reduce
from html import unescape
from itertools import groupby

from django.conf import settings
from pytils.translit import translify, detranslify

from intranet.search.abovemeta import forms
from intranet.search.abovemeta.request import Request
from intranet.search.abovemeta.utils import has_digits, translate, string_to_int
from intranet.search.core.sources.idm.utils import replace_special_chars
from intranet.search.core.sources.st.utils import get_group_attrs_for_factor_by_uid
from intranet.search.core.sources.utils import swap_layout
from intranet.search.core.query import parse_query
from intranet.search.core.query import ast, ast_factors
from intranet.search.core.utils import get_kps, QueryDict

NUMBER_LOOKUP_MAP = {
    forms.ATTR_LOOKUP_GTE: ast.AttrGreaterEqualSearch,
    forms.ATTR_LOOKUP_GT: ast.AttrGreaterSearch,
    forms.ATTR_LOOKUP_LT: ast.AttrLessSearch,
    forms.ATTR_LOOKUP_LTE: ast.AttrLessEqualSearch,
    forms.ATTR_LOOKUP_EQ: ast.AttrSearch,
}


class RequestBuilder:
    request_stub = ast.Text('%request%')
    tvm_client_name = None
    request_builder_fields = ['formula', 'wizard_rules', 'wizard_params', 'revisions', 'groupings',
                              'attributes', 'qtree', 'sorted', 'sorted_order', 'p']

    def __init__(self, search, state):
        self.search = search
        self.params = QueryDict()
        self.state = state
        self.qtree = self.search.get('qtree') or ast.Text('')

        # мягкие ограничения, которые просто дописываются к тексту запроса
        self.soft_restriction = None
        # жесткие ограничения, которые передаются отдельным параметром restrict
        self.hard_restriction = None

    @property
    def name(self):
        return self.search['name']

    @property
    def endpoint(self):
        return self.search['search_settings']['endpoint']

    def get_request(self, type='search', **kwargs):
        ticket = self.state.tvm_service_tickets.get(self.tvm_client_name, '')
        return Request(
            url=self.endpoint.url(query=self.build_params()),
            type_=type,
            name=self.name,
            headers={
                settings.TVM2_SERVICE_HEADER: ticket,
            },
            **kwargs
        )

    def process_params(self):
        raise NotImplementedError

    def compile_query(self):
        self.params['text'] = self.qtree.to_string()

        # Отправляем все дополнительные фильтры запроса через параметр template.
        # Это нужно, чтобы опечаточник исправлял опечатки только в тексте запроса,
        # ну и вообще так правильнее.
        # Документация: https://doc.yandex-team.ru/Search/saas/saas-overview/concepts/searching.html#searching__template-dd
        if self.params.get('template'):
            template = self.params['template'].format(qtree=self.request_stub)
        else:
            template = self.request_stub

        if self.soft_restriction:
            if self.params['text']:
                template = ast.Constr(template, self.soft_restriction)
            else:
                template = self.soft_restriction

        # FIXME не уверена, что это нужно, но у нас везде в compile_query переопределяется qtree
        self.qtree = template
        self.params['template'] = template.to_string()

        if self.hard_restriction:
            self.params['restrict'] = '(%s)' % self.hard_restriction.to_string()

    def parse_query(self):
        if self.params.get('qtree') is not None:
            self.qtree = self.params.pop('qtree')
        elif self.params.get('text'):
            self.qtree = parse_query(self.params.pop('text'), 5)

    def build_params(self):
        for field in self.request_builder_fields:
            if self.search['search_settings'].get(field) is not None:
                self.params[field] = self.search['search_settings'][field]

        if self.search.get('template'):
            self.params['template'] = self.search['template']

        self.parse_query()
        self.process_params()
        self.compile_query()
        return self.params


class CommonSearch(RequestBuilder):
    SAAS_TIMEOUT = 5  # таймаут при запросах в саас по умолчанию
    tvm_client_name = 'saas'

    def process_params(self):
        self.apply_filters()
        self.apply_format()
        self.apply_revisions()
        self.apply_sort()
        self.apply_attr_limit()
        self.apply_formula()
        self.apply_wizard()
        self.apply_properties()
        self.apply_context_grouping()
        self.apply_pruncount()
        self.apply_timeout()
        self.apply_user_info()
        self.apply_relev_info()
        self.apply_features()
        self.apply_misspell()
        self.apply_request_marker()
        self.apply_softness()

    def apply_revisions(self):
        revisions = self.params.pop('revisions')

        revisions_by_service = {}
        for service, service_revisions in groupby(revisions, lambda x: x['service']):
            revisions_by_service[service] = list(service_revisions)
        self.params['service'] = ','.join(revisions_by_service)

        if len(revisions_by_service) == 1:
            # Временный фикс для ISEARCH-5152
            # Если делаем запрос только в один сервис, то используем только 'kps' без префикса
            param_tmpl = 'kps'
        else:
            param_tmpl = '{service}_kps'

        for service, revisions in revisions_by_service.items():
            param_name = param_tmpl.format(service=service)
            self.params[param_name] = ','.join(str(get_kps(r['id'])) for r in revisions)

    def get_astfilter(self, zone_name, expr, quoted=True):
        """
        Хелперная функция помогающая создать ast для поиска в зоне или атрибуте
        >>> get_astfilter('public', '1').to_string() == 'public:1'
        """
        if isinstance(expr, str):
            if quoted:
                expr = ast.OrderedText(expr)
            else:
                expr = ast.Text(expr)

        return ast.AttrSearch(ast.Keyword(zone_name), expr)

    def apply_properties(self, lst=None):
        default = {'doc_id', 'doc_native_id', 'doc_revision',
                       'doc_source', 'doc_url', 'doc_index', 'key',
                       'title', 'clickUrl'}

        for language in self.state.priority_languages:
            default.add(f'doc_snippet_{language}')

        default.update(lst or [])
        self.params.setlist('gta', default)

    @property
    def can_search_public_wiki_pages(self):
        """
        Пермишшен "search_public_wiki_pages" даёт пользователю право
        искать "публичные" вики-страницы
        https://st.yandex-team.ru/ISEARCH-6332
        """
        return 'search_public_wiki_pages' in self.state.user_permissions

    def apply_acl(self, user=None):
        """ учитываем права доступа """
        user = user or self.state.user_identifier

        # добавляет в запрос ограничения по правам доступа
        public = self.get_astfilter('public', '1')

        if not self.can_search_public_wiki_pages:
            # Note: в идеале проверять, что s_type != wiki,
            # но по словам @tmalikova в saas != (<>) работает не очень хорошо
            doc_or_posts = ast.Or(
                self.get_astfilter('s_type', 'doc'),
                self.get_astfilter('s_type', 'post'),
            )
            public = ast.And(public, doc_or_posts)

        if not user:
            self.apply_restriction(public)
        else:
            groups = ast.Or.join(
                self.get_astfilter('acl_groups_whitelist', str(group))
                for group in self.state.groups
            )
            users = self.get_astfilter('acl_users_whitelist', user)

            self.apply_restriction(ast.Or.join(filter(None, (public, groups, users))))

    def apply_features(self):
        """ Применяем к поиску дополнительные параметры, определяемые через фичи
        """
        # для ISEARCH-5037 помимо strictsyntax учимся перенастраивать
        # фичи на определенные колдунщики
        if self.state.feature_enabled('search_wizhosts'):
            self.params['wizhosts'] = self.state.feature_value('search_wizhosts')

    def apply_sort_by_attr(self, attr):
        self.params['how'] = attr

    def apply_user_info(self):
        self.params['uuid'] = self.state.saas_uid
        self.params['client-ip'] = self.state.user_ip
        if not self.state.feature_enabled('skip_experiments'):
            self.params['test_buckets'] = self.state.ab_info.get('buckets') or ''

    def apply_sort(self):
        sort = self.params.pop('sorted', None)
        if self.params.pop('sorted_order', None) == 'asc':
            self.params['asc'] = True
        if sort in ['date', 'updated']:
            self.apply_sort_by_attr('updated')
        else:
            self.apply_sort_by_attr(sort or 'rlv')

    def apply_hold_request(self):
        '''поиск в найденном'''
        if self.params.get('holdreq'):
            request = parse_query(unescape(self.params['holdreq']), 5)
            self.apply_constraint(request)

    def apply_grouping(self, field, per_page, per_group):
        self.params['g'] = f'1.{field}.{per_page}.{per_group}.-1'

    def apply_empty_grouping(self, per_page):
        # если не нужна группировка - то указываем это первым нулем
        self.params['g'] = f'0..{per_page}.1.-1'

    def apply_search_filter_grouping(self, field):
        self.params.update({'g': f'1.{field}.10000000.1.-1.0.0.-1.rlv.0.count'})

    def apply_format(self):
        self.params.update({'format': 'json', 'snip': 'in_json_report=yes'})

    def apply_attr_limit(self):
        self.apply_relev('attr_limit', '10000000')

    def apply_relev(self, key, value):
        self.params.update({'relev': f'{key}={value}'})

    def apply_constraint(self, constraint):
        if self.soft_restriction:
            self.soft_restriction = ast.Constr(self.soft_restriction, constraint)
        else:
            self.soft_restriction = constraint

    def apply_restriction(self, restriction):
        if self.hard_restriction:
            self.hard_restriction = ast.And(self.hard_restriction, restriction)
        else:
            self.hard_restriction = restriction

    def clean_facets(self):
        # Игнорируем фасеты, которые не определены для этого поиска
        self.state.facets = {
            k: v
            for k, v in self.state.facets.items()
            if k in self.state.scope_facets
        }

    def apply_filters(self):
        # FIXME сделать какой-нибудь метод is_organic
        if self.name != 'search_results':
            return

        if self.state.scope_facets:
            self.clean_facets()
            self.apply_facets_names(self.state.scope)
            self.apply_facets(self.state.facets)

        self.apply_attr_lookup()
        self.apply_zones()

    def apply_facets(self, facets):
        filters = []
        for name, values in facets.items():
            filter_name = name if name.startswith('s_') else 's_' + name
            filters.extend(self.get_astfilter(filter_name, v) for v in values)
        if filters:
            self.apply_constraint(ast.AndSoft.join(filters))

    def apply_attr_lookup(self):
        """ Добавляет к поиску фильтры по числовым атрибутам
        """
        lookup = self.state.attrs
        filters = []
        for lookup_type in lookup:
            node = NUMBER_LOOKUP_MAP.get(lookup_type)
            if not node:
                continue
            for name, value in lookup[lookup_type].items():
                filter_name = name if name.startswith('i_') else 'i_' + name
                filters.append(node(ast.Keyword(filter_name), ast.OrderedText(value)))
        if filters:
            self.apply_constraint(ast.AndSoft.join(filters))

    def apply_zones(self):
        filters = []
        if self.state.feature_enabled('zone_fulltext_filter'):
            zone_text_node = ast.ParenthesizedText
        else:
            zone_text_node = ast.OrderedText

        for name, query in self.state.zones.items():
            zone_name = name if name.startswith('z_') else 'z_' + name
            filters.append(ast.ZoneSearch(ast.Keyword(zone_name), zone_text_node(query)))
        if filters:
            self.apply_constraint(ast.AndSoft.join(filters))

    def apply_facets_names(self, scope):
        """ Используется для получения информации по нативным фасетам при
        обращении к поиску

        Эта функция только определяет список фасетов которые мы хотим получить,
        затем передает его в _apply_facets_names
        """
        facet_names_list = self.state.scope_settings['facets']
        self._apply_facets_names(facet_names_list)

    def _apply_facets_names(self, facet_names_list):
        """Преобразует имена фасетов в вид нужный для saas и добавляет
        соответствующие cgi параметры
        """
        facets = []
        qi = []

        for name, options in facet_names_list.items():
            sname = name if name.startswith('s_') else 's_' + name
            qi.append('facet_' + sname)

            # добавлем ограничение на количество отданных фасетов
            top = options.get('top_number')

            if top:
                sname += ':%s' % top

            facets.append(sname)

        self.params.setlist('qi', qi)
        self.params.update({'facets': ','.join(facets)})

    def apply_formula(self):
        formula = self.params.pop('formula')
        if formula:
            self.apply_relev('formula', formula)

    def apply_wizard(self):
        wizard_rules = self.params.pop('wizard_rules')
        if wizard_rules:
            rules = []
            keywords = {'on', 'off'}
            for rule in wizard_rules.strip().split(','):
                if rule in keywords:
                    rules.append(rule)
                elif rule.startswith('-'):
                    rules.append('off:' + rule[1:])
                else:
                    rules.append('on:' + rule)
            self.params.setlist('rwr', rules)

        wizard_params = self.params.pop('wizard_params')
        if wizard_params:
            self.params.update(wizard_params)

    def apply_context_grouping(self):
        groupings = self.params.pop('groupings', [])
        for group in groupings:
            if group.get('field'):
                self.apply_grouping(group['field'], group['per_page'], group['per_group'])
            else:
                self.apply_empty_grouping(group['per_page'])

    def apply_timeout(self, timeout=None):
        sec = 1000000
        if not timeout:
            timeout = string_to_int(self.state.feature_value('saas_timeout')) or self.SAAS_TIMEOUT
        self.params['timeout'] = timeout * sec

    def apply_haha(self):
        # ускоряет ответ благодаря тому, что не возвращает реальные данные
        self.params['haha'] = 'da'
        # помечаем такие запросы роботными, чтобы они не попадали в логи пользовательских сессий
        self.params['robot'] = 'da'

    def apply_pruncount(self, count=9000):
        pron = self.params.getlist('pron')
        for p in pron:
            if p.startswith('pruncount'):
                pron.remove(p)

        self.apply_pruning('pruncount%s' % count)

    def apply_pruning(self, pron):
        self.params.update(pron=pron)

    def apply_relev_info(self):
        self.params.update({
            'dbgrlv': 'da',
            'fsgta': '_JsonFactors',
            'relev': 'all_factors',
        })

    def apply_misspell(self):
        """ Применение политики опечаточника.
        Документация: https://doc.yandex-team.ru/Search/saas/saas-overview/concepts/spellcheck.html)
        """
        self.params['msp'] = 'no' if self.state.nomisspell else 'try_at_first'

    def apply_request_marker(self):
        """ Добавляет маркеры запроса: это атрибуты, которые ни на что не влияют,
        но они будут залогированы в access_log сааса и мы сможем по ним отличить
        разные вертикали, колдунщики, саджест и т.п.
        """
        self.params['intrasearch-scope'] = self.state.scope
        self.params['intrasearch-zone'] = self.name or ''
        self.params['intrasearch-reqid'] = self.state.request_id
        self.params['intrasearch-referer-host'] = self.state.referer_hostname
        suggest_session_id = self.state.req_headers.get('X-Suggest-Session-Id')
        if suggest_session_id:
            self.params['intrasearch-suggest-session-id'] = suggest_session_id

    def apply_softness(self):
        """ Добавляет к запросу параметр softness, который нужен для того,
        чтобы искалось не полное совпадение со всеми словами запроса.
        Принимает значение от 0 до 100, где 0 - искать все слова, 100 - хотя бы одно слово.
        Дока: https://doc.yandex-team.ru/Search/y-server-manual/concepts/query-zones-and-attributes-search.html
        """
        feature_name = f'{self.state.scope}_{self.name}_softness'
        feature_value = self.state.feature_value(feature_name)
        if feature_value is not None:
            softness = ast.AttrSearch('softness', feature_value)
            template = self.params.get('template') or ast.Placeholder('qtree')
            self.params['template'] = ast.AndSoft(template, softness)


class WikiSearch(CommonSearch):

    def clean_facets(self):
        """ Дополнительно чистит фасеты по кластеру: определяет уровень кластера
        и чистит длину значения, если пытаются фильтровать по уровню глубже, чем мы умеем
        """
        cleaned_facets = defaultdict(list)
        for key, values in self.state.facets.items():
            for value in values:
                if self._is_cluster_facet(key):
                    bits = value.split('/')
                    level = len(bits)
                    size = settings.ISEARCH_WIKI_CLUSTER_FACET_DEPTH
                    if bits[0] == 'users':
                        level -= 1
                        size += 1

                    if level > settings.ISEARCH_WIKI_CLUSTER_FACET_DEPTH:
                        level = settings.ISEARCH_WIKI_CLUSTER_FACET_DEPTH

                    key = self._get_cluster_name(level)
                    value = '/'.join(bits[:size])

                cleaned_facets[key].append(value)

        self.state.facets = cleaned_facets
        super().clean_facets()

    def apply_facets_names(self, scope):
        cluster_facets = [(k, v) for k, v in self.state.facets.items() if self._is_cluster_facet(k)]
        if cluster_facets:
            # Если есть фильтрация по кластеру, показываем фильтр по самому глубокому из них
            # (в норме у нас вообще один фильтр по кластеру),
            # а также фильтр по кластеру на уровень выше и на уровень ниже (если возможно)
            cluster_facet_name, cluster_facet_value = max(cluster_facets)
            # уровень кластера определяем из названия фасета
            current_level = int(cluster_facet_name[len('cluster'):])

            levels_to_show = []
            for delta in (-1, 0, 1):
                level = current_level + delta
                if 0 < level <= settings.ISEARCH_WIKI_CLUSTER_FACET_DEPTH:
                    levels_to_show.append(level)
        else:
            # Если нет фильтрации по кластеру, то показываем только один верхний уровень
            current_level = 1
            levels_to_show = [current_level]

        facet_names = self.state.scope_facets
        facet_names_to_apply = {k: v for k, v in facet_names.items() if not self._is_cluster_facet(k)}
        for level in levels_to_show:
            name = self._get_cluster_name(level)
            facet_names[name]['name'] = {
                self.state.language: self._get_wiki_cluster_name(level, current_level)
            }
            facet_names_to_apply[name] = facet_names[name]

        self._apply_facets_names(facet_names_to_apply)

    def _get_cluster_name(self, level):
        return 'cluster%s' % level

    def _is_cluster_facet(self, name):
        return name.startswith('cluster')

    def _get_wiki_cluster_name(self, cluster_level, current_level):
        if cluster_level == current_level:
            label = 'CLUSTER_IN'
        elif cluster_level < current_level:
            label = 'CLUSTER_UP'
        else:
            label = 'CLUSTER_DOWN'
        return translate(label, self.state)

    def process_params(self):
        super().process_params()

        self.apply_hold_request()
        self.apply_acl()


class BiWikiSearch(WikiSearch):
    """ Билдер для поиска по Вики для бизнеса
    """
    can_search_public_wiki_pages = True  # в b2b все могут искать публичные вики-страницы

    def apply_acl(self, user=None):
        # В b2b вики все равно ищем по логину, несмотря на то, что всюду в b2b acl по uid
        super().apply_acl(user=self.state.user)


class MetaSearch(WikiSearch):
    def process_params(self):
        super().process_params()
        self.apply_empty_grouping(10)
        self.apply_show_request_options()
        self.apply_full_doc_filter()
        self.apply_doc_lang_filter()

    def apply_show_request_options(self):
        """ Включает для каждого источника метапоиска в выдачу query который
        был в него отправлен

        Объяснения в тикете https://st.yandex-team.ru/SAAS-1954

        """
        qi = self.params.getlist('qi')
        qi.append('rty_request')
        self.params.setlist('qi', qi)

        self.params['rearr'] = 'scheme_Local/RTYProperties/DumpRequest=1'

    def apply_full_doc_filter(self):
        if self.state.feature_enabled('only_full_doc'):
            only_full = ast.Or.join([
                ast.And(
                    self.get_astfilter('s_type', 'doc'),
                    self.get_astfilter('i_is_part', 0)
                ),
                self.get_astfilter('s_type', 'post'),
                self.get_astfilter('s_type', 'wiki'),
            ])
            self.apply_restriction(only_full)

    def apply_doc_lang_filter(self):
        doc_lang = ast.Or.join([
            ast.And(
                self.get_astfilter('s_type', 'doc'),
                ast.Or.join([
                    self.get_astfilter('s_doc_lang', self.state.language),
                    self.get_astfilter('s_doc_lang', 'all'),
                ]),
            ),
            self.get_astfilter('s_type', 'post'),
            self.get_astfilter('s_type', 'wiki'),
        ])
        self.apply_restriction(doc_lang)


class DocMetaSearch(CommonSearch):
    def process_params(self):
        super().process_params()
        self.apply_acl()
        self.apply_hold_request()
        self.apply_full_doc_filter()
        self.apply_doc_lang_filter()

    def apply_full_doc_filter(self):
        if self.state.feature_enabled('only_full_doc'):
            only_full = self.get_astfilter('i_is_part', 0)
            self.apply_restriction(only_full)

    def apply_doc_lang_filter(self):
        doc_lang = ast.Or.join([
            self.get_astfilter('s_doc_lang', self.state.language),
            self.get_astfilter('s_doc_lang', 'all'),
        ])
        self.apply_restriction(doc_lang)


class PeopleSearch(CommonSearch):
    def process_params(self):
        super().process_params()
        self.apply_acl()
        self.apply_context_grouping()
        self.apply_user_factors()
        self.params['maxpassages'] = 0

    def apply_user_access_restrictions(self):
        restrictions = self.get_astfilter('s_login', self.state.user)
        if self.state.user_departments_access:
            groups = ast.Or.join(
                self.get_astfilter('i_group_id', str(group))
                for group in self.state.user_departments_access
            )
            restrictions = ast.Or(restrictions, groups)
        self.apply_restriction(restrictions)

    def apply_acl(self, user=None):
        if self.state.feature_enabled('hide_dismissed'):
            self.apply_restriction(
                ast.Or(
                    self.get_astfilter('i_is_dismissed', '0'),
                    self.get_astfilter('i_is_memorial', '1'),
                )
            )
        if self.state.feature_enabled('enable_people_acl'):
            if not self.state.user_has_full_staff_access:
                self.apply_user_access_restrictions()

    def apply_user_factors(self):
        if self.state.staff_id:
            is_me_factor = ast_factors.Eq('#group_staff_id', self.state.staff_id)
            self.apply_relev('calc', f'USER_people_is_me:{is_me_factor.to_string()}')

        # атрибут state: (группировочный атрибут, фактор)
        insetany_factors = {
            'user_departments': ('department', 'in_my_department'),
            'user_parent_departments': ('department', 'in_parent_department'),
            'user_services': ('service', 'in_my_service'),
            'user_frequently_searched_people': ('staff_id', 'frequently_searched'),
            'user_recently_searched_people': ('staff_id', 'recently_searched'),
        }

        for state_attr, (group_attr, factor_suffix) in insetany_factors.items():
            state_data = getattr(self.state, state_attr, None)
            if state_data:
                factor = ast_factors.InSetAny(f'#group_{group_attr}', state_data)
                self.apply_relev('calc', f'USER_people_{factor_suffix}:{factor.to_string()}')


class TrackerSearch(CommonSearch):
    factor_prefix = 'st'

    def apply_sort(self):
        sort = self.params.pop('sorted', None)
        if sort in ['date', 'updated']:
            self.params['how'] = 'updated'

    def apply_user_factors(self):
        self.apply_queue_st_factor()
        self.apply_factors_by_uid()

    def apply_factors_by_uid(self):
        """ Добавляем пользовательские факторы по автору и исполнителю
        """
        for field in ('author', 'assignee'):
            parts = []
            for name, value in get_group_attrs_for_factor_by_uid(self.state.user_uid, field):
                parts.append(ast_factors.Eq(f'#group_{name}', value))
            factor = ast_factors.And(*parts)

            self.apply_relev('calc', 'USER_{prefix}_is_{field}:{factor}'.format(
                prefix=self.factor_prefix, field=field, factor=factor.to_string()))

    def apply_queue_st_factor(self):
        if not self.state.user_queues:
            return

        user_queue = ast_factors.InSet('#group_queue', self.state.user_queues)
        self.apply_relev('calc', 'USER_{prefix}_queue:{factor}'.format(
            prefix=self.factor_prefix, factor=user_queue.to_string()))

    def process_params(self):
        super().process_params()

        self.apply_acl()
        self.apply_user_factors()
        self.apply_context_grouping()


class BisearchTrackerSearch(TrackerSearch):
    factor_prefix = 'tracker'


class YasenSearch(TrackerSearch):

    def apply_filters(self):
        super().apply_filters()
        self.apply_constraint(self.get_astfilter('s_queue', 'YASEN'))


class StatSearch(CommonSearch):
    def process_params(self):
        super().process_params()


class AtSearch(CommonSearch):
    posts_filtred = False

    def process_params(self):
        super().process_params()

        # если нет уже группировок (могут быть добавлены фасетами)
        if not self.params.get('g'):
            self.apply_grouping('thread_id_grp', 10, 10)

    def apply_search_filter_grouping(self, field):
        super().apply_search_filter_grouping(field)

        if not self.posts_filtred:
            self.apply_constraint(self.get_astfilter('s_type', 'post'))
            self.posts_filtred = True


class SuggestRequestBuilderMixin:
    """ Миксин для всех реквест билдеров саджеста
    """
    SAAS_TIMEOUT = 2

    @property
    def layout_changers(self):
        layout_changers = [
            lambda x: x,
            partial(swap_layout, orig='ru'),
            partial(swap_layout, orig='en'),
        ]

        if self.state.feature_enabled('translify_suggest'):
            layout_changers += [translify, detranslify]
        return layout_changers

    @property
    def zone_suggest_feature(self):
        return f'{self.name}_zone_suggest'

    @property
    def is_zone_suggest_enabled(self):
        return self.state.feature_enabled(self.zone_suggest_feature)

    @property
    def use_misspell(self):
        return self.state.feature_enabled(f'{self.name}_suggest_use_misspell')

    def compile_query(self):
        # В саджесте не используем саасные шаблоны для поиска
        if self.soft_restriction:
            self.qtree = ast.Constr(self.qtree, self.soft_restriction)

        self.params['text'] = self.qtree.to_string()
        if self.hard_restriction:
            self.params['restrict'] = '(%s)' % self.hard_restriction.to_string()

    def apply_suggest(self):
        nodes = []
        for text_node in self.qtree.filter_text_nodes():
            nodes.append(self.get_astfilter('s_suggest', text_node.text + '*', True))
        return reduce(ast.And, nodes)

    def apply_layer_restrictions(self):
        """ Добавляет ограничения, заданные для конкретного типа саджеста
        """
        query = self.search['search_settings'].get('layer_query', None)
        if query:
            self.apply_constraint(query)

    def process_empty_query(self):
        # для ускорения поиска при наличии layer_query и пустом запросе просто ищем по layer_query,
        # а не по url:"*" << layer_query
        query = self.search['search_settings'].pop('layer_query', None)
        if query:
            self.qtree = query
        else:
            # При url:"*" не считаются пользовательские факторы, например.
            # Поэтому используем s_suggest:"*"
            self.qtree = self.get_astfilter('s_suggest', '*')

    def process_suggest_query(self):
        if not self.is_zone_suggest_enabled:
            self.qtree = ast.Or(self.apply_suggest(), self.qtree)
            return self.qtree

        # ISEARCH-6251: Замена s_suggest на поиск по зоне.
        # Включается через фичу, доступны 2 режима:
        # 1. default - поисковая строка "таис мали" превращается в "таис мали* | tais mali* | ..."
        # 2. by_word - "таис мали" превращается в "таис* & мали* | tais* & mali* | ..."
        by_word = self.state.feature_value(self.zone_suggest_feature) == 'by_word'
        self.params['wizextra'] = 'usewildcards=da'
        qtree = self.qtree
        if by_word:
            words = (ast.Text(n.text + '*') for n in self.qtree.filter_text_nodes())
            qtree = reduce(ast.And, words)

        # Пробуем не передавать в саас все варианты исправления запроса,
        # а использовать встроенный в него опечаточник: он сам применит исправление,
        # если по исходному запросу ничего не будет найдено
        if self.use_misspell:
            layouts = {qtree.to_string()}
            self.state.nomisspell = False
            self.apply_misspell()
        else:
            layouts = {l(qtree.to_string()) for l in self.layout_changers}

        nodes = (ast.Text(l if by_word else l + '*') for l in layouts)
        self.qtree = reduce(ast.Or, nodes)
        return self.qtree

    def process_params(self):
        super().process_params()

        if self.state.text:
            self.process_suggest_query()
        else:
            self.process_empty_query()

        self.apply_layer_restrictions()

    def apply_request_marker(self):
        super().apply_request_marker()
        self.params['intrasearch-suggest-version'] = self.state.version


class Suggest(SuggestRequestBuilderMixin, CommonSearch):
    """ Основной реквест билдер для саджестов
    """


class DocSuggest(SuggestRequestBuilderMixin, DocMetaSearch):
    """ Реквест билдер для саджестов по документации.
    """


class QueuesSuggest(SuggestRequestBuilderMixin, TrackerSearch):
    """ Реквест билдер для саджестов по очередям стартрека.
    """
    def apply_user_factors(self):
        self.apply_queue_st_factor()


class BisearchQueuesSuggest(SuggestRequestBuilderMixin, BisearchTrackerSearch):
    """
    Реквест билдер для саджестов по очередям стартрека в b2b.
    """
    def apply_user_factors(self):
        self.apply_queue_st_factor()


class IssuesSuggest(SuggestRequestBuilderMixin, TrackerSearch):
    """ Реквест билдер для саджестов по тикетам стартрека.
    """


class BisearchIssuesSuggest(SuggestRequestBuilderMixin, BisearchTrackerSearch):
    """ Реквест билдер для саджестов по тикетам стартрека в b2b.
    """


class PeopleSuggest(SuggestRequestBuilderMixin, PeopleSearch):

    def process_suggest_query(self):
        search_text = self.qtree.to_string()
        has_space_symbol = ' ' in search_text

        if self.is_zone_suggest_enabled:
            nodes = [super().process_suggest_query()]
        else:
            nodes = [
                self.apply_suggest(),
                self.get_astfilter('z_people_name', search_text, quoted=False),
            ]
            if has_space_symbol:
                nodes.append(self.get_astfilter('s_suggest', search_text + '*', quoted=True))

        # если пользователи вводят цифры с пробелами, то это может быть
        # телефон или номер машины, стоит поискать их и без пробелов
        if has_space_symbol and has_digits(search_text):
            text = search_text.replace(' ', '')
            if self.is_zone_suggest_enabled:
                nodes.append(ast.Text(text + '*'))
            else:
                nodes.append(self.get_astfilter('s_suggest', text))

        self.qtree = reduce(ast.Or, nodes)
        return self.qtree

    def apply_user_access_restrictions(self):
        if self.state.version == 0:
            super().apply_user_access_restrictions()


class IDMSuggest(Suggest):
    """ Реквест билдер для саджестов по idm
    """
    def process_params(self):
        super().process_params()
        self.apply_user_factors()

    def apply_user_factors(self):
        if self.state.user_departments:
            dep_factor = ast_factors.InSet('#group_department', self.state.user_departments)
            self.apply_relev('calc', 'USER_idm_department:%s' % dep_factor.to_string())

            if self.state.user_parent_departments:
                all_deps = self.state.user_parent_departments + self.state.user_departments
                nearest_dep = ast_factors.Or(
                    ast_factors.InSet('#group_department', self.state.user_parent_departments),
                    ast_factors.InSet('#group_parent_department', all_deps)
                )
            else:
                nearest_dep = ast_factors.InSet('#group_parent_department', self.state.user_departments)
            self.apply_relev('calc', 'USER_idm_nearest_department:%s' % nearest_dep.to_string())

        if self.state.user_services:
            services_factor = ast_factors.InSet('#group_service', self.state.user_services)
            self.apply_relev('calc', 'USER_idm_service:%s' % services_factor.to_string())

    def apply_suggest(self):
        query = super().apply_suggest()

        search_text = self.qtree.to_string()
        if ' ' in search_text:
            # если в запросе есть пробел, то ищем не только по совпадению начала каждого слова,
            # но и по всей фразе целиком: "новые ро*"
            query = ast.Or(query, self.get_astfilter('s_suggest', search_text + '*', True))

        # ноды типа conductor.maps_load_renderer_carparks хотим находить
        # по "maps_load_renderer_carp" и по "renderer_carp". Для этого основная магия происходит
        # в индексаторе (см. sources.idm.rolenodes), а здесь нужно заменить все нужные
        # разделители пробелами
        sub_query = replace_special_chars(search_text)
        if sub_query != search_text:
            query = ast.Or(query, self.get_astfilter('s_suggest', sub_query + '*', True))

        return query


class WikiSuggest(SuggestRequestBuilderMixin, WikiSearch):
    """ Реквест билдер для Wiki
    """


class AtSuggest(SuggestRequestBuilderMixin, AtSearch):
    """ Реквест билдер для саджеста по этушке
    """


class CommitsSearch(CommonSearch):
    def process_params(self):
        super().process_params()
        self.apply_pruncount(40000)


class MldescriptionSearch(CommonSearch):
    def process_params(self):
        super().process_params()
        if not self.state.feature_enabled('ml_all_types'):
            self.apply_constraint(self.get_astfilter('s_email_type', 'ml'))


class CandidateSearch(CommonSearch):
    """ Поиск по кандидатам Фемиды
    """
    request_stub = ast.ParenthesizedText('%request%')

    def _get_name_from_query(self):
        if not self.state.people_names:
            return None

        name = ' '.join(v for v in self.state.people_names[0].values() if v)
        return name

    def apply_acl(self):
        if self.state.feature_enabled('femida_force_filter_duplicates'):
            # временно явно добавляем фемиде фильтр по дубликатам, как только
            # они поддержат эту функциональность у себя - фильтр и фичу можно будет убрать
            self.apply_restriction(self.get_astfilter('i_is_duplicate', '0'))

        if self.state.acl_restrictions:
            self.apply_restriction(self.state.acl_restrictions)

    def parse_query(self):
        super().parse_query()
        if not self.qtree or self.state.feature_enabled('femida_search_everywhere'):
            return

        name_from_query = self._get_name_from_query()
        if name_from_query and self.state.feature_enabled('femida_use_name_zone'):
            search_zone = 'z_femida_name'
            self.qtree = ast.Text(name_from_query)
        else:
            search_zone = 'z_base'

        self.params['template'] = ast.ZoneSearch(ast.Keyword(search_zone), ast.Placeholder('qtree'))

    def process_params(self):
        super().process_params()
        self.apply_acl()

    def apply_facets_names(self, scope):
        """ Заглушка для кандидатов. Значениями из фасетов фемида не пользуется,
        поэтому не будем замедлять поиск их подсчетом
        """
        if self.state.feature_enabled('femida_show_facets'):
            super().apply_facets_names(scope)


class EquipmentSearch(CommonSearch):
    def apply_base_location_factors(self):
        if self.state.feature_enabled('equipment_rebase_location'):
            # Новая локация задаётся в формате <id офиса>[-<id этажа>], несуществующие id игнорируются
            # Соответствие этажа офису не проверяется (это независимые факторы), так что можно задавать любые комбинации
            # id офисов и этажей можно посмотреть, например, тут: https://staff-api.yandex-team.ru/v3/offices
            rebase_location = self.state.feature_value('equipment_rebase_location').split('-', 1)
            base_office = rebase_location[0]
            base_floor = rebase_location[1] if len(rebase_location) == 2 else None
        else:
            base_office = self.state.user_base_office
            base_floor = self.state.user_base_floor

        if base_office:
            base_office_factor = ast_factors.Eq('#group_office', base_office)
            self.apply_relev('calc', 'USER_equipment_base_office:%s' % base_office_factor.to_string())
        if base_floor:
            base_floor_factor = ast_factors.Eq('#group_floor', base_floor)
            self.apply_relev('calc', 'USER_equipment_base_floor:%s' % base_floor_factor.to_string())

    def process_params(self):
        super().process_params()
        self.apply_base_location_factors()


class MLSearch(RequestBuilder):
    tvm_client_name = 'ml'

    @property
    def endpoint(self):
        return settings.ISEARCH['api']['ml']['search']

    def build_params(self):
        self.params.update({
            'uid': self.state.user_uid,
            'mdb': 'pg',
            'request': self.search['search_settings']['qtree'].to_string(),
            'page': self.search['search_settings'].get('p', 0),
        })
        return self.params


class ServicesSearch(CommonSearch):

    def process_params(self):
        super().process_params()
        self.apply_user_factors()

    def apply_user_factors(self):
        if self.state.staff_id:
            is_member_factor = ast_factors.InSetAny('#group_member', [self.state.staff_id])
            self.apply_relev('calc', f'USER_plan_is_member:{is_member_factor.to_string()}')


class ServicesSuggest(SuggestRequestBuilderMixin, ServicesSearch):
    is_zone_suggest_enabled = True
