import logging
from socket import error as socket_error
import time
from tornado.httpclient import AsyncHTTPClient, HTTPError
from tornado import gen
from urllib.parse import urlencode, urlsplit, parse_qs, urlunsplit

from django.conf import settings


timeout_log = logging.getLogger('timeouts')
logger = logging.getLogger(__name__)


AsyncHTTPClient.configure("tornado.curl_httpclient.CurlAsyncHTTPClient")


class Requester:
    def __init__(self, retries=3):
        self.requests = []
        self.client = AsyncHTTPClient()
        self.retries = retries

    @gen.coroutine
    def fetch(self, request, state=None):
        response = None
        retries = request.retries or self.retries

        for attempt in range(retries):
            req = request.get_tornado_request()
            started = time.time()
            try:
                response = yield self.client.fetch(req)
                code = response.code
            except HTTPError as exc:
                # Сюда попадает все что не 2xx, при этом при
                # code==599 response==None
                logger.exception(str(exc), extra={'state': state})
                response = exc.response
                code = exc.code
            except socket_error as exc:
                logger.exception(str(exc), extra={'state': state})
                code = exc.errno
            finally:
                request_time = time.time() - started

            if request.name in settings.ISEARCH_ABOVEMETA_NO_LOG:
                parts = list(urlsplit(request.url))
                query = parse_qs(parts[3])
                new_query = {a: a.upper() for a in settings.ISEARCH_ABOVEMETA_NO_LOG[request.name]}
                query.update(new_query)

                parts[3] = urlencode(query, True)
                request.url = urlunsplit(parts)

            logger.log(
                logging.INFO if code == 200 else logging.ERROR,
                '%s.%s: %d - %f sec (%s %s)',
                request.type,
                request.name,
                code,
                response.request_time if response else request_time,
                request.method,
                request.url,
                extra={'state': state, 'stack': True})

            req_data = {
                'request': request,
                'response': response,
                'code': code,
                'attempt': attempt,
                'request_time': request_time,
                'started': started,
            }

            self.requests.append(req_data)

            if code in (599, 504):
                timeout_log.info("Timeout (%d) %d [%s] %s",
                                 attempt + 1, code, req.method, request.url,
                                 extra={'state': state})
            elif code >= 500:
                continue
            else:
                break

        raise gen.Return(response)
