from datetime import datetime
from logging import getLogger
from urllib.parse import urlparse

from django.conf import settings
from django.utils import timezone

from intranet.search.core.sources.utils import (
    date_isoformat,
    get_document_url,
    timestamp_to_utc_date,
    truncate_chars,
)
from intranet.search.core.utils.http import absolute_url, cut_sheme

log = getLogger(__name__)


def get_work_phone(snippet):
    work_phone = (lambda p: [i['value'] for i in p if i['type'] == 'work'])(
        snippet.get('phones', []))
    work_phone_value = str(work_phone[0]) if work_phone else None
    return work_phone_value


def suggest_people_serializer(snippet, doc):
    cars = [{'number': c['plate'], 'model': c['model']} for c in snippet.get('cars', [])]
    return {
        'title': '{} {}'.format(snippet['name']['first'], snippet['name']['last']),
        'url': 'https://staff.yandex-team.ru/%s' % snippet['login'],
        'id': snippet['login'],
        'fields': [
            {'type': 'staff_id', 'value': snippet.get('id')},
            {'type': 'uid', 'value': snippet.get('uid')},
            {'type': 'login', 'value': snippet['login']},
            {'type': 'name', 'value': snippet['name']},
            {'type': 'phone', 'value': get_work_phone(snippet)},
            {'type': 'department_name', 'value': snippet.get('department', {}).get('dep_name')},
            {'type': 'position', 'value': snippet['position']},
            {'type': 'is_dismissed', 'value': snippet['is_dismissed']},
            {'type': 'affiliation', 'value': snippet.get('affiliation')},
            {'type': 'is_memorial', 'value': snippet['is_memorial']},
            {'type': 'is_robot', 'value': snippet['is_robot']},
            {'type': 'bicycles', 'value': snippet.get('bicycles', [])},
            {'type': 'cars', 'value': cars},
            {'type': 'gaps', 'value': snippet.get('gaps', [])},
        ]
    }


def suggest_people_lite_serializer(snippet, doc):
    return {
        'title': '{} {}'.format(snippet['name']['first'], snippet['name']['last']),
        'url': 'https://staff.yandex-team.ru/%s' % snippet['login'],
        'id': snippet['login'],
        'fields': [
            {'type': 'staff_id', 'value': snippet.get('id')},
            {'type': 'uid', 'value': snippet.get('uid')},
            {'type': 'login', 'value': snippet['login']},
            {'type': 'is_dismissed', 'value': snippet['is_dismissed']},
            {'type': 'affiliation', 'value': snippet.get('affiliation')},
            {'type': 'is_memorial', 'value': snippet['is_memorial']},
            {'type': 'is_robot', 'value': snippet['is_robot']},
        ]
    }


def suggest_groups_serializer(snippet, doc):
    data = {
        'title': snippet['title'],
        'url': snippet['url'],
        'id': str(snippet['id']),
        'fields': [
            {'type': 'slug', 'value': snippet['slug']},
            {'type': 'type', 'value': snippet['type']},
            {'type': 'service', 'value': (snippet['service'] or {}).get('slug')},
            {'type': 'service_id', 'value': (snippet['service'] or {}).get('id')},
            {'type': 'department_id', 'value': (snippet['department'] or {}).get('id')},
            {'type': 'department', 'value': (snippet['department'] or {}).get('slug')},
            {'type': 'affiliation_counters', 'value': snippet.get('affiliation_counters', {})},
        ]
    }
    return data


def suggest_doc_serializer(snippet, doc):
    data = {
        'title': snippet['title'],
        'id': snippet['url'],
        'url': snippet['url'],
        'fields': [
            {'type': 'description', 'value': snippet.get('description', '')},
            {'type': 'passages', 'value': doc.get_passages()},
            {'type': 'product', 'value': snippet.get('product', '')},
        ]
    }
    return data


def suggest_nav_serializer(snippet, doc):
    return {
        'title': snippet.get('title'),
        'url': snippet['href'],
        'id': snippet['slug'],
        'fields': [
            {'type': 'slug', 'value': snippet['slug']},
            {'type': 'abc', 'value': snippet['abc']},
            {'type': 'service_name', 'value': snippet['service_name']},
        ]
    }


def suggest_services_serializer(snippet, doc):
    return {
        'title': snippet['name'],
        'url': doc.url,
        'id': snippet['id'],
        'fields': [
            {'type': 'slug', 'value': snippet['slug']},
            {'type': 'status', 'value': snippet['status']},
            {'type': 'owner', 'value': snippet['person']},
        ],
    }


def suggest_queues_serializer(snippet, doc):
    return {
        'title': snippet['title'],
        'url': snippet['url'],
        'id': snippet['key'],
        'fields': []
    }


def suggest_issues_serializer(snippet, doc):
    url = snippet.get('url') or doc.url
    try:
        snippet['created'] = date_isoformat(timestamp_to_utc_date(snippet['created']))
    except Exception:
        log.exception(
            'Cannot convert tracker timestamp to datetime: created=%s',
            snippet.get('created'),
        )

    return {
        'id': snippet['key'],
        'title': snippet['summary'],
        'url': url,
        'created': snippet.get('created'),
        'author': snippet.get('author'),
        'fields': [],
    }


def suggest_ml_serializer(snippet, doc):
    return {
        'title': snippet['name'],
        'url': snippet['url'],
        'id': snippet['email'],
        'fields': [
            {'type': 'description', 'value': snippet.get('info', '')}
        ]
    }


def suggest_idm_rolenodes_serializer(snippet, doc):
    data = {
        'title': snippet['title'],
        'url': '',
        'id': '/{}{}'.format(snippet.get('system'), snippet['slug_path']),
        'fields': []
    }
    for f in ('system', 'slug', 'slug_path', 'parent_path', 'aliases', 'help'):
        data['fields'].append({'type': f, 'value': snippet.get(f)})
    return data


def suggest_idm_subjects_serializer(snippet, doc):
    if snippet.get('object_type') == 'groups':
        data = suggest_idm_groups_serializer(snippet, doc)
    else:
        data = suggest_idm_users_serializer(snippet, doc)
    data['fields'].append({'type': 'object_type', 'value': snippet['object_type']})
    return data


def suggest_idm_groups_serializer(snippet, doc):
    data = {
        'title': snippet['title'],
        'url': snippet['url'],
        'id': snippet['object_id'],
        'fields': []
    }
    for f in ('slug', 'state', 'type', 'ancestors'):
        data['fields'].append({'type': f, 'value': snippet.get(f)})
    return data


def suggest_idm_users_serializer(snippet, doc):
    data = {
        'title': '{} {}'.format(snippet['first_name'], snippet['last_name']),
        'url': snippet['url'],
        'id': snippet['login'],
        'fields': []
    }
    for f in ('is_active', 'department_name', 'first_name', 'last_name', 'email'):
        data['fields'].append({'type': f, 'value': snippet.get(f)})
    return data


def suggest_invite_serializer(snippet, doc):
    data = {
        'title': snippet.get('name'),
        'url': snippet.get('map_url'),
        'id': snippet.get('id'),
        'fields': []
    }
    for f in ('city_id', 'city_name', 'office_id', 'office_name', 'floor_id',
              'floor_name', 'phone', 'name_alternative', 'name_exchange', 'video_conferencing'):
        data['fields'].append({'type': f, 'value': snippet.get(f)})
    return data


def suggest_clinics_serializer(snippet, doc):
    data = {
        'title': snippet.get('name'),
        'url': snippet.get('url'),
        'id': snippet.get('id'),
        'fields': []
    }
    for f in ('address', 'phones', 'status', 'categories', 'working_hours', 'metro', 'region'):
        data['fields'].append({'type': f, 'value': snippet.get(f)})
    return data


def suggest_wiki_serializer(snippet, doc):
    if not snippet.get('url'):
        snippet['url'] = doc.url
    for bcrumb in snippet.get('breadcrumbs', []):
        bcrumb['url'] = absolute_url(bcrumb['url'], snippet['url'])

    return {
        'id': snippet.get('url'),
        'url': snippet.get('url'),
        'title': snippet.get('title'),
        'breadcrumbs': snippet.get('breadcrumbs'),
        'fields': []
    }


def suggest_atushka_serializer(snippet, doc):
    if not snippet.get('url'):
        snippet['url'] = doc.url

    if not snippet.get('title') and snippet.get('type') == 'post':
        snippet['title'] = cut_sheme(snippet['url'])

    try:
        snippet['updated'] = date_isoformat(timestamp_to_utc_date(snippet.get('updated')))
    except KeyError:
        pass
    except Exception:
        log.exception(
            'Cannot convert atushka timestamp to datetime: updated=%s',
            snippet.get('updated')
        )

    return {
        'id': snippet.get('url'),
        'url': snippet.get('url'),
        'title': snippet.get('title'),
        'author': snippet.get('author'),
        'updated': snippet.get('updated'),
        'club': snippet.get('club'),
        'fields': []
    }


def search_st_serializer(snippet, doc):
    if not snippet.get('url'):
        snippet['url'] = doc.url

    for field in ('updated', 'created'):
        try:
            # переводим таймстамп в даты в isoformat.
            # делается здесь, а не в самом трекере, потому что не можем
            # взять и всё переиндексировать
            snippet[field] = date_isoformat(timestamp_to_utc_date(snippet[field]))
        except Exception:
            log.exception(
                'Cannot convert tracker timestamp to datetime: %s=%s',
                field, snippet.get('field'),
            )
            continue
    snippet['title'] = snippet['summary']
    snippet['tags'] = snippet['tags'] or []
    if snippet['description'] == 'None':
        # чтобы не переиндексировать весь стратрек при выводе заменяем None на пустую строку
        snippet['description'] = ''
    else:
        snippet['description'] = truncate_chars(snippet['description'], 250)
    return snippet


def search_mldescription_serializer(snippet, doc):
    if not snippet.get('url'):
        snippet['url'] = doc.url
    snippet['description'] = snippet.get('info', '')
    snippet['description_pure'] = snippet.get('info_pure', '')
    return snippet


def search_at_serializer(snippet, doc):
    if not snippet.get('url'):
        snippet['url'] = doc.url

    if not snippet.get('title') and snippet.get('type') == 'post':
        snippet['title'] = cut_sheme(snippet['url'])

    for field in ('updated', 'published'):
        try:
            # переводим таймстамп в даты в isoformat.
            # делается здесь, а не в самом трекере, потому что не можем
            # взять и всё переиндексировать
            snippet[field] = date_isoformat(timestamp_to_utc_date(snippet[field]))
        except KeyError:
            continue
        except Exception:
            log.exception('Cannot convert at timestamp to datetime: %s=%s',
                          field, snippet.get('field'))
            continue
    for field in ('replies', 'likes', 'dislikes'):
        snippet[field] = snippet.get(field, 0)
    return snippet


def search_at_clubs_serializer(snippet, doc):
    # приводим id к строке, потому что json не умеет очень длинные инты
    snippet['id'] = str(snippet['id'])
    if not snippet.get('url'):
        snippet['url'] = doc.url

    for field in ('first_post_date', 'last_post_date'):
        if not snippet[field]:
            continue
        try:
            d = datetime.strptime(snippet[field], '%Y-%m-%d %H:%M:%S').replace(tzinfo=timezone.utc)
            snippet[field] = date_isoformat(d)
        except Exception:
            log.exception('Cannot convert at.clubs timestamp to datetime: %s=%s',
                          field, snippet.get('field'))
            continue
    return snippet


def search_people_serializer(snippet, doc):
    if not snippet.get('url'):
        snippet['url'] = doc.url

    # сделано здесь, а не сразу в индексаторе для обратной совместимости с xml.
    # при переходе фронта полностью на json можно убрать лишние данные из сниппета в индекса
    for car in snippet['cars']:
        if car.get('plate') == car.get('plate_pure'):
            car['plate'] = max_highlighted(car.get('number')) or car.get('plate')
        car['number'] = car.get('plate') or car['number'] or ''
        car['number_pure'] = car.get('plate_pure') or ''
        car.pop('plate', None)
        car.pop('plate_pure', None)

    if snippet.get('department', {}).get('url'):
        url = get_document_url('department').format(url=snippet['department']['url'])
        snippet['department']['url'] = url

    return snippet


def max_highlighted(values_list, highlight_mark=settings.HILITE_START_TAG):
    if not values_list:
        return
    return max(values_list, key=lambda a: a.count(highlight_mark))


def search_wiki_serializer(snippet, doc):
    if not snippet.get('url'):
        snippet['url'] = doc.url

    try:
        snippet['is_public'] = bool(int(snippet.pop('public', 0) or 0))
    except ValueError:
        log.warning('Cannot convert "public" to int', extra={'context': {'snippet': snippet}})
        snippet['is_public'] = bool(int(snippet.pop('public_pure', 0) or 0))

    try:
        raw_date = snippet.get('modtime_pure') or snippet['modtime']
        d = datetime.strptime(raw_date, '%Y-%m-%d %H:%M:%S').replace(tzinfo=timezone.utc)
        snippet['modtime'] = date_isoformat(d)
    except Exception:
        log.exception('Cannot convert wiki timestamp to datetime: modtime=%s', snippet.get('field'))
    snippet.pop('modtime_timestamp', None)
    snippet.pop('modtime_pure', None)

    for bcrumb in snippet.get('breadcrumbs', []):
        bcrumb['url'] = absolute_url(bcrumb['url'], snippet['url'])

    return snippet


def search_plan_services_serializer(snippet, doc):
    if not snippet.get('url'):
        snippet['url'] = doc.url

    for contact in snippet['contacts']:
        if 'tracker' in contact['type']:
            contact['title'] = 'Очередь %s' % contact['title']
            contact['title_pure'] = 'Очередь %s' % contact['title_pure']
    return snippet


def search_yasensearch_st_serializer(snippet, doc):
    """
    Сериалайзер для поиска по Ясеню (сниппеты берутся из индекса трекера)

    Почему тут всё не так, как везде?
    Это попытка сделать новый универсальный сниппет, который бы автоматически красиво
    верстался на фронтенде.
    Здесь черновой вариант бэкенда, сделанный в лоб. Если формат зайдёт в идеале нужно
    написать какой-то универсальный форматтер/сериалайзер, который приводит сниппеты к нужному виду
    (структура описана здесь: https://st.yandex-team.ru/ISEARCH-6561#5f5fdd2fddc9d406de44db87).

    Если вы читаете этот коммент через год+ после его написания, а код всё такой же стрёмный,
    то можете найти kuznecov@ и пожурить :)
    """

    result = {
        'type': 'generic',
        'url': doc.url,
        'title': None,
        'image': None,
        'main': None,
        'more': None,
    }
    issue_key = snippet.get('key_pure') or snippet['key']
    if snippet['additional'] is None:
        snippet['additional'] = {}

    result['title'] = [
        {
            'type': 'st-link',
            'theme': 'black',
            'key': issue_key,
            'value': snippet['key'],
            'resolved': snippet['resolution']['id'] != 'none',
            'url': doc.url,
        },
        ': ',
        {
            'type': 'link',
            'key': issue_key,
            'title': snippet.get('summary_pure', snippet['summary']),
            'value': snippet['summary'],
            'url': doc.url,
        },
        ' ',
        snippet['status']['name'],
    ]

    images = [a for a in snippet['attachments'] or [] if 'image' in a['mimetype']]
    if images:
        main_image = images[0]
        main_image_id = snippet['additional'].get('thumbnailId')
        if main_image_id:
            main_image_candidate = [i for i in images if str(i['id']) == str(main_image_id)]
            if main_image_candidate:
                main_image = main_image_candidate[0]
        thumbs = [i['url'] for i in images if i['url'] != main_image['url']][:3]
        result['image'] = {
            'type': 'st-link',
            'key': issue_key,
            'url': doc.url,
            'value': [{
                'type': 'main-image',
                'url': doc.url,
                'main': main_image['url'],
                'thumbs': thumbs,
                'more': len(images) - len(thumbs),
            }],
        }

    result['main'] = [
        {
            'type': 'key-value',
            'items': [
                {
                    'key': 'Сервис',
                    'value': ', '.join(s['name'] for s in snippet['services']) if snippet['services'] else None,
                },
                {
                    'key': 'Правообладатель',
                    'value': snippet['additional'].get('copyrightHolder'),
                },
                {
                    'key': 'Источник',
                    'value': {
                        'type': 'link',
                        'url': snippet['additional']['sourceDescription'],
                        'value': (
                            urlparse(snippet['additional']['sourceDescription']).hostname
                            or snippet['additional']['sourceDescription']
                        ),
                    } if snippet['additional'].get('sourceDescription') else None
                },
                {
                    'key': 'Автор',
                    'value': [
                        {
                            'type': 'user',
                            'name': snippet['author'].get('name_pure', snippet['author']['name']),
                            'login': snippet['author'].get('login_pure', snippet['author']['login']),
                        },
                        ' ',
                        {
                            'type': 'date',
                            'value': date_isoformat(timestamp_to_utc_date(snippet['created'])),
                        },
                    ],
                },
                {
                    'key': 'Теги',
                    'value': {
                        'type': 'list',
                        'items': snippet['tags'],
                    } if snippet['tags'] else None,
                },
                {
                    'key': 'Компоненты',
                    'value': {
                        'type': 'list',
                        'items': [comp['name'] for comp in snippet['components']],
                    } if snippet['components'] else None,
                },
            ],
        },
    ]

    license_period = []
    for field in ('start', 'end'):
        if snippet['additional'].get(field):
            day = datetime.strptime(snippet['additional'][field], '%Y-%m-%d')
            license_period.append(day.strftime('%d.%m.%Y'))

    result['more'] = {
        'text': {
            'collapse': 'Свернуть подробности о лицензии',
            'expand': 'Развернуть подробности о лицензии',
        },
        'value': {
            'type': 'key-value',
            'items': [
                {
                    'key': 'Лицензия предоставлена',
                    'value': snippet['additional'].get('license'),
                },
                {
                    'key': 'Срок',
                    'value': ' - '.join(license_period) if license_period else None,
                },
                {
                    'key': 'Территория',
                    'value': snippet['additional'].get('countryNew'),
                },
                {
                    'key': 'Способы использования',
                    'value': snippet['additional'].get('typeOfContract'),
                },
            ],
        },
    }
    if not any(i['value'] for i in result['more']['value']['items']):
        result['more'] = None

    return result
