import json
from base64 import b64decode
from logging import getLogger

from django.conf import settings
from django.utils.encoding import force_bytes

from intranet.search.abovemeta.request import Request
from intranet.search.abovemeta.steps.base import HttpStep, timeouts


log = getLogger(__name__)


class ABInfoStep(HttpStep):
    """ Стадия получения информации о том в какие ab эксперименты
    попадает пользователь
    """
    def get_request(self, state):
        endpoint = settings.ISEARCH['api']['uaas']
        headers = self.get_headers(state)

        return Request(
            url=endpoint.url(query=self.get_query(state)),
            headers=headers,
            type_='ab',
            name='uaas',
            retries=2,
            request_timeout=timeouts['uaas'],
        )

    def get_headers(self, state):
        cookie = state.req_headers.get('Cookie')
        if isinstance(cookie, bytes):
            cookie = cookie.decode('ascii', errors='replace')
        if isinstance(cookie, str):
            cookie = cookie.encode('ascii', errors='replace')
        headers = {
            'X-Forwarded-For-Y': state.user_ip,
            'User-Agent': state.req_headers.get('User-Agent'),
            'Cookie': cookie,  # TODO: после починки залипания - оставить только нужные куки
        }
        return headers

    def get_query(self, state):
        query = {
            'text': state.text,
            'uuid': state.saas_uid,
            'test-id': state.test_id,
        }
        return query

    def process_response(self, state, response):
        if not response:
            log.error('UAAS cannot get response', extra={'state': state})
            return
        if response.code != 200:
            log.error('UAAS response error: %s, %s', response.code, response.body,
                      extra={'state': state})
            return

        log.info('UAAS got data: %s for user %s', str(response.headers), state.user_identifier,
                 extra={'state': state})
        buckets = response.headers.get('X-Yandex-ExpBoxes')  # <testid1>,0,<bucket1>;<testid2>,0,<bucket2>;...
        crypted_buckets = response.headers.get('X-Yandex-ExpBoxes-Crypted')  # для experiments в яндекс.метрику
        try:
            for encoded_flags in response.headers.get('X-Yandex-ExpFlags', '').split(','):
                if not encoded_flags:
                    continue

                data = json.loads(b64decode(force_bytes(encoded_flags)))
                data = data[0] if isinstance(data, list) else data
                if data.get('HANDLER') != 'INTRASEARCH':
                    continue
                exp_features = data.get('CONTEXT', {}).get('INTRASEARCH', {}).get('FEATURES', {})
                state.update_features({k.lower(): v for k, v in exp_features.items()})
        except Exception:
            log.warning('UAAS cannot parse exp flags: %s', response.headers.get('X-Yandex-ExpFlags'),
                        extra={'state': state})
        state.ab_info = {
            'buckets': buckets,
            'crypted_buckets': crypted_buckets,
        }
