import os

import logging
from django.conf import settings
from tornado import gen

from intranet.search.abovemeta import errors
from intranet.search.abovemeta.request import Request
from intranet.search.abovemeta.steps.base import HttpStep, Step
from intranet.search.abovemeta.utils import split_to_int_list

USER_ORGS_ATTR = '1017'
USER_LANG_ATTR = '34'
CLOUD_UID_MIN = 9 * (10 ** 15)
CLOUD_UID_MAX = CLOUD_UID_MIN + 10 ** 12

log = logging.getLogger(__name__)


def is_cloud_uid(uid):
    return CLOUD_UID_MIN < uid < CLOUD_UID_MAX


class AuthStep(HttpStep):
    def get_future(self, state, requester):
        if (settings.TVM2_SERVICE_HEADER in state.req_headers and
                settings.TVM2_USER_HEADER not in state.req_headers and
                state.tvm_auth_service in settings.TVM2_WEREWOLFING_IDS):
            state.auth_method = 'tvm_werewolf'
            future = gen.Future()
            werewolf_result = {
                'uid': {
                    'value': settings.TVM2_WEREWOLFING_USER_UID
                },
                'login': settings.TVM2_WEREWOLFING_USER_LOGIN,
                'attributes': {USER_LANG_ATTR: 'ru'}
            }
            future.set_result(werewolf_result)
            return future

        return super().get_future(state, requester)

    def get_request(self, state):
        endpoint = settings.ISEARCH['api']['blackbox']
        ip = state.user_ip or state.remote_ip
        headers = endpoint.headers()
        headers.update({
            'X-Ya-Service-Ticket': state.tvm_service_tickets.get('blackbox', ''),
        })

        if settings.TVM2_USER_HEADER in state.req_headers:
            state.auth_method = 'tvm'
            state.tvm_auth_user_ticket = state.req_headers[settings.TVM2_USER_HEADER]
            query = {
                'method': 'user_ticket',
                'user_ticket': state.tvm_auth_user_ticket,
            }
        else:
            if 'Authorization' in state.req_headers:
                state.auth_method = 'oauth'
                query = {
                    'method': 'oauth',
                    'userip': ip,
                }
                headers['Authorization'] = state.req_headers['Authorization']
            else:
                state.auth_method = 'cookie'
                query = {
                    'method': 'sessionid',
                    'sessionid': state.user_session_id,
                    'host': state.host,
                    'userip': ip,
                }
            query['get_user_ticket'] = 'yes'

        attributes = [USER_ORGS_ATTR, USER_LANG_ATTR]
        query['attributes'] = ','.join(attributes)

        return Request(
            url=endpoint.url(query=query),
            headers=headers,
            type_='auth',
            name='auth',
        )

    def process_response(self, state, response):
        if isinstance(response, dict):
            parsed = response
            self.set_data(state, parsed)
        else:
            parsed = super().process_response(state, response)
        return parsed

    def parse_response(self, state, response):
        data = super().parse_response(state, response)
        # user_ticket присылает данные в другом виде - списком пользователей
        if 'users' in data:
            data = data['users'][0]
        return data

    def set_data(self, state, auth_data):
        status = auth_data.get('status') or auth_data.get('exception')
        if status and status['value'] not in ('VALID', 'NEED_RESET'):
            msg = 'Auth failed: status={}, {}'.format(status['value'], auth_data.get('error'))
            log.error('Auth failed: status=%s, %s', status['value'], auth_data.get('error'),
                      extra={'state': state})
            state.set_error(errors.ERROR_AUTH, msg)
            return

        state.user = auth_data['login'].split('@')[0].lower()
        state.user_uid = auth_data['uid']['value']
        state.user_identifier = state.user
        state.user_language = auth_data.get('attributes', {}).get(USER_LANG_ATTR)
        if not state.tvm_auth_user_ticket:
            state.tvm_auth_user_ticket = auth_data.get('user_ticket')

    def set_error(self, state, response):
        try:
            msg = response.error
        except AttributeError:
            msg = 'Auth no response'
        log.error('Auth failed: %s', msg, extra={'state': state})
        state.set_error(errors.ERROR_AUTH, msg)


class DevAuthStep(Step):
    def process_response(self, state, response):
        self.set_data(state, {})

    def set_data(self, state, auth_data):
        state.user = os.environ['USER']
        state.user_uid = os.environ.get('USER_UID', 123)
        state.user_identifier = state.user


if settings.YENV_TYPE == 'development':
    AuthStep = DevAuthStep  # noqa: RedefinedWhileUnused


class BisearchAuthStep(AuthStep):

    def get_user_uid(self, state):
        backend_application = settings.ISEARCH['tvm']['clients']['source']['app_id']
        try:
            uid = int(state.req_headers.get('X-UID'))
        except (TypeError, ValueError):
            return
        if state.req_headers.get('X-Cloud-UID') or state.tvm_auth_service == backend_application:
            return uid
        return uid if is_cloud_uid(uid) else None

    def get_cloud_uid(self, state):
        return state.req_headers.get('X-Cloud-UID')

    def get_future(self, state, requester):
        uid = self.get_user_uid(state)
        cloud_uid = self.get_cloud_uid(state)
        backend_application = settings.ISEARCH['tvm']['clients']['source']['app_id']
        cloud_applications = list(settings.ISEARCH['tvm'].get('cloud_applications', []))
        cloud_applications.append(backend_application)

        if (uid or cloud_uid) and state.tvm_auth_service in cloud_applications:
            future = gen.Future()
            future.set_result({'uid': {'value': str(uid)}, 'cloud_uid': cloud_uid, 'login': ''})
            state.is_cloud_user = True
            state.auth_method = 'cloud'
            return future
        else:
            return super().get_future(state, requester)

    def set_data(self, state, auth_data):
        super().set_data(state, auth_data)
        state.user_identifier = state.user_uid
        state.cloud_uid = auth_data.get('cloud_uid')

        attributes = auth_data.get('attributes') or {}
        log.info('USER_ORGS_ATTR=%s', attributes.get(USER_ORGS_ATTR), extra={'state': state})

        try:
            user_orgs = split_to_int_list(attributes.get(USER_ORGS_ATTR))
        except Exception:
            log.exception('Cannot parse user organizations: %s', attributes.get(USER_ORGS_ATTR),
                          extra={'state': state})
            user_orgs = []

        if user_orgs:
            if state.requested_org_directory_id:
                if state.requested_org_directory_id in user_orgs:
                    state.user_has_org = True
                    state.org_directory_id = state.requested_org_directory_id
                else:
                    # Если пользователь запрашивает организацию, которой нет в атрибуте блэкбокса,
                    # то, возможно, он её внешний админ. Проверим это в DirectoryStep.
                    log.warning('User does not have organization %s. Available organizations: %s',
                                state.requested_org_directory_id, user_orgs,
                                extra={'state': state})
                    state.user_has_org = False
            elif len(user_orgs) == 1:
                state.user_has_org = True
                state.org_directory_id = user_orgs[0]
            else:
                log.error('User has several organizations: %s', user_orgs, extra={'state': state})
                state.set_error(errors.ERROR_SPECIFY_ORGANIZATION)
        else:
            state.org_directory_id = state.requested_org_directory_id
            state.user_has_org = state.is_cloud_user
