import json
import logging

from tornado import gen
from tornado.httpclient import HTTPResponse

from django.conf import settings
from django.core.cache import caches

api_urls = settings.ISEARCH['abovemeta']
timeouts = settings.ISEARCH['abovemeta']['timeouts']

log = logging.getLogger(__name__)


class Step:
    def get_future(self, state, requester):
        """
        Переопределяется в наследниках, возвращает таску с основной
        логикой степа. Не переопределяется для тех у кого нет хождений по http
        """
        moment = gen.Future()
        moment.set_result(None)
        return moment

    @gen.coroutine
    def execute(self, state, requester):
        """ Метод который запускает выполнение таски из self.get_future """

        if not state.errors:
            res = yield self.get_future(state, requester)
            self.process_response(state, res)

    def process_response(self, state, response):
        """ Переопределяется в наследниках. По умолчанию не делает ничего. """


class CompoundStep(Step):
    def __init__(self, steps=None, **kwargs):
        super().__init__(**kwargs)
        self.steps = steps or []

    def get_steps(self, state=None):
        # Нужна для таких наследников, которые сами генерируют свои степы на основе state
        return self.steps

    def get_future(self, state, requester):
        return self.execute(state, requester)

    def process_response(self, state, response):
        pass


class HttpStep(Step):
    def get_request(self, state):
        """ Метод для тасок которые ходят по http,
        они возвращают request.Request или None
        """
        raise NotImplementedError

    def get_future(self, state, requester):
        """
        Для тасок которые просто ходят по http за данными этоот метод
        переопределять не нужно

        """
        request = self.get_request(state)
        self.update_headers(state, request)
        return requester.fetch(request, state)

    def update_headers(self, state, request):
        if 'X-Request-Id' not in request.headers:
            request.headers['X-Request-Id'] = state.request_id

    def parse_response(self, state, response):
        return json.load(response.buffer)

    def process_response(self, state, response):
        if not response or response.code != 200:
            self.set_error(state, response)
        else:
            try:
                parsed = self.parse_response(state, response)
            except Exception:
                log.exception('Error while parsing response %s', response.request.url,
                              extra={'state': state})
                self.set_error(state, response)
            else:
                self.set_data(state, parsed)
                return parsed

    def set_data(self, state, response):
        raise NotImplementedError

    def set_error(self, state, response):
        """ Метод вызывается в случае, если произошла ошибка в запросе (!=200)
        """
        pass


class CachedHttpStep(HttpStep):
    """ Http степ с кешированием результата
    """
    cache_timeout = getattr(settings, 'ISEARCH_ABOVEMETA_HTTP_CACHE_TIMEOUT', 300)

    def __init__(self):
        super().__init__()
        self.cache = caches['local_memcached']

    def get_cache_key(self, state):
        raise NotImplementedError()

    def get_future(self, state, requester):
        cached_data = self.cache.get(self.get_cache_key(state))
        if cached_data:
            future = gen.Future()
            future.set_result(cached_data)
            return future
        else:
            return super().get_future(state, requester)

    def process_response(self, state, response):
        if not response or isinstance(response, HTTPResponse):
            parsed = super().process_response(state, response)
            self.cache.set(self.get_cache_key(state), parsed, self.cache_timeout)
        else:
            parsed = response
            self.set_data(state, parsed)
        return parsed


class FoundDocsConditionalHttpStepMixin:
    """
    Миксин, дающий возможность выполнять шаг только если что-то найдено
    """
    search = None
    index = None

    def need_to_do_step(self, state, docs):
        return bool(docs)

    def get_future(self, state, requester):
        found_docs = self.get_found_docs(state)
        if not self.need_to_do_step(state, found_docs):
            future = gen.Future()
            future.set_result(None)
            return future

        request = self.get_request(state, found_docs)
        self.update_headers(state, request)
        return requester.fetch(request, state)

    def get_found_docs(self, state):
        search_result = None

        for result in state.searches.values():
            if result['search'] == self.search and result['index'] == self.index:
                search_result = result

        if not search_result or not search_result.get('parsed'):
            return []

        return search_result['parsed'].get_docs()


class ParallelStep(CompoundStep):
    @gen.coroutine
    def execute(self, state, requester):
        steps = self.get_steps(state)
        if not state.errors and steps:
            responses = yield [step.get_future(state, requester) for step in steps]

            for response, step in zip(responses, steps):
                step.process_response(state, response)

            self.process_response(state, responses)


class ChainStep(CompoundStep):
    @gen.coroutine
    def execute(self, state, requester):
        data = []
        steps = self.get_steps(state)

        for step in steps:
            if state.errors:
                break
            response = yield step.get_future(state, requester)
            data.append(response)
            step.process_response(state, response)

        self.process_response(state, data)


class SleepStep(Step):
    def __init__(self, sleep, type_='', **kwargs):
        super().__init__(**kwargs)
        self.sleep = sleep
        self.type = type_

    def get_future(self, state, requester):
        return gen.sleep(self.sleep)

    def process_response(self, state, data):
        from datetime import datetime
        print(f'hi from sleep {self.sleep} {self.type} {datetime.now()}')
