import logging
from collections import defaultdict

from ylog.context import log_context

from intranet.search.core.query import parse_query
from intranet.search.abovemeta import errors

from intranet.search.abovemeta.search_settings import prepare_search
from intranet.search.abovemeta.decider import select
from intranet.search.abovemeta.steps.base import Step

log = logging.getLogger(__name__)


class BaseDeciderStep(Step):
    """ Базовый класс шага принятия решения о тому, в какие индексы сааса идти
    """

    def find_searches(self, state, response):
        raise NotImplementedError()

    def process_response(self, state, response):
        self.find_searches(state, response)
        if not state.searches:
            log.error('Cannot find searches for %s', state.scope, extra={'state': state})
            self.set_error(state, 'No searches found')

    def set_error(self, state, message):
        state.set_error(errors.ERROR_SEARCH_DECIDER, message)


class DeciderStep(BaseDeciderStep):
    def get_future(self, state, requester):
        return select(state, requester)

    def find_searches(self, state, response):
        """
        Есть две зоны:
            - search_results - органическая выдача, правило по ней срабатывает всегда безусловно
            - wizard - колдунщики, могут быть на 1 или 5 позиции, могут быть на обоих или только
            на одной позиции, может не быть вообще
        """
        targets_by_zone = self.get_sorted_targets_by_zones(response)

        search_results_spell = targets_by_zone['search_results'][0]
        search_results_data = self.get_search_data(state, search_results_spell, 'search_results')
        if search_results_data:
            state.searches['search_results'] = search_results_data
        search_results_rating = search_results_spell['context']['rating']

        if not targets_by_zone['wizard']:
            return

        wizard_positions = [1, 5]
        if state.feature_enabled('ml_wizard_ranking'):
            # если вес колдунщика меньше, чем вес первой позиции в органике, то не ставим
            # его на первое место
            first_wizard_rating = targets_by_zone['wizard'][0]['context']['rating']
            if first_wizard_rating < search_results_rating:
                wizard_positions = [5]

        for position, wizard in zip(wizard_positions, targets_by_zone['wizard']):
            data = self.get_search_data(state, wizard, 'wizard', position)
            if data:
                state.searches[data['name']] = data

    def get_sorted_targets_by_zones(self, response):
        targets_by_zone = defaultdict(list)
        for triggered, (rule, context) in response:
            if triggered:
                targets_by_zone[rule.zone].append({'rule': rule, 'context': context})

        for spells in targets_by_zone.values():
            spells.sort(key=lambda s: s['context']['rating'], reverse=True)
        return targets_by_zone

    def get_search_data(self, state, spell, zone, position=1):
        name = f'{zone}_{position}' if position > 1 else zone
        rule = spell['rule']
        settings = prepare_search(state, rule.search, rule.collection, rule.formula)
        if not settings:
            log.error(
                'No revision found. search %s, index: %s, organization_id: %s',
                rule.search, rule.collection, state.org_directory_id,
                extra={'state': state}
            )
            return

        additional = {
            'groupings': rule.groupings,
            'attributes': rule.attributes,
            'qtree': spell['context']['qtree'],
        }

        if name == 'search_results':
            if state.sorted:
                additional['sorted'] = state.sorted
            if state.sorted_order:
                additional['sorted_order'] = state.sorted_order
            if state.page:
                additional['p'] = state.page

        settings.update(additional)

        data = {
            'name': name,
            'wizard_name': spell['rule'].name,
            'text': spell['context']['qtree'].to_string(),
            'position': position,
            'rating': spell['context']['rating'],
            'search_settings': settings,
            'search': rule.search,
            'collection': rule.collection,
            'index': settings['index'],
        }
        return data


class SuggestDeciderStep(BaseDeciderStep):
    def add_layer(self, state, name, search=None, collection=None, **kwargs):
        if state.debug:
            log.info(
                'SuggestDeciderStep.add_layer arguments, name: %s, search: %s, collection: %s',
                name, search, collection
            )
        settings = prepare_search(state, search, collection, 'suggest')
        if not settings:
            log.error(
                'Not revisions found for search: %s, collection: %s, organization_id: %s',
                search, collection, state.org_directory_id, extra={'state': state}
            )
            return

        settings['groupings'] = [{'field': '', 'per_page': kwargs['per_page']}]

        settings['qtree'] = state.qtree
        settings['p'] = int(kwargs.get('page') or 0)

        if kwargs.get('query'):
            settings['layer_query'] = parse_query(kwargs['query'], 5)

        with log_context(settings=settings):
            if state.debug:
                log.info('SuggestDeciderStep.add_layer settings')

        data = {
            'name': name,
            'text': state.text,
            'search_settings': settings,
            'search': search,
            'collection': collection,
            'index': settings['index'],
        }

        state.searches[name] = data

    def find_searches(self, state, response):
        with log_context(layers=state.layers):
            if state.debug:
                log.info('SuggestDeciderStep.find_search layers')

        for layer, settings in state.layers.items():
            self.add_layer(state, layer, **settings)
