import json
import logging

from django.conf import settings

from intranet.search.core.query import parse_query
from intranet.search.abovemeta import errors
from intranet.search.abovemeta.errors import ERROR_UNKNOWN
from .base import HttpStep, ParallelStep

log = logging.getLogger(__name__)
CURRENT_USER_FACET = 'me()'


class SimpleSearchStep(HttpStep):
    def __init__(self, search, *args, **kwargs):
        self.search = search
        super().__init__(*args, **kwargs)

    def process_response(self, state, response):
        if not response:
            self.set_error(state, response)
            return

        try:
            parsed = self.parse_response(state, response)
        except Exception:
            log.exception('Error while parsing response %s', response.request.url,
                          extra={'state': state})
            self.set_error(state, response)
        else:
            if response.code != 200 and self.search['name'] == 'search_results':
                state.set_error(errors.WARNING_INCOMPLETE_RESPONSE)
                log.error('Incomplete saas response: %s, status_code=%s',
                          response.request.url, response.code, extra={'state': state})
            self.set_data(state, parsed)
            return parsed

    def parse_response(self, state, response):
        context = {
            'view': 'user',
            'domain': state.domain,
        }
        services = {rev['service'] for rev in self.search['search_settings']['revisions']}
        cls = self.search['search_settings']['response_class']
        return cls(
            raw_response=json.load(response.buffer),
            languages=state.priority_languages,
            services=services,
            context=context,
        )

    def is_valid_response(self, state, parsed):
        if settings.ISEARCH['abovemeta'].get('skip_revision_check'):
            log.info('Feature skip_revision_check enabled', extra={'state': state})
            return True
        # Проверяем, что саас прислал нам именно те ревизии, которые мы запросили.
        founded_revisions = parsed.revisions
        requested_revisions = {str(rev['id']) for rev in self.search['search_settings']['revisions']}

        if founded_revisions - requested_revisions:
            log.error('Got redundant revisions from SaaS: requested=%s, founded=%s',
                      requested_revisions, founded_revisions, extra={'state': state})
            self.set_error(state, None)
            return False
        return True

    def get_request(self, state):
        request_builder_cls = self.search['search_settings']['request_builder']
        self.prepare_facets(state)
        request_builder = request_builder_cls(self.search, state)
        request = request_builder.get_request(type='search')

        self.search['qtree'] = request_builder.qtree
        self.search['endpoint'] = request.url

        return request

    def prepare_facets(self, state):
        """ Подготовка фасетов для поиска
        """
        # заменяем me() на имя пользователя в фильтрах по пользователю
        for facet_name in state.user_facets:
            try:
                idx = state.facets.get(facet_name, []).index(CURRENT_USER_FACET)
            except ValueError:
                continue
            else:
                state.facets[facet_name][idx] = state.user_identifier

    def process_facets(self, state, parsed):
        facets = parsed.facets

        for key, values in state.facets.items():
            for value in values:
                facet_data = {'value': value, 'doccount': 0}
                if key not in facets:
                    facets[key] = {value: facet_data}
                elif value not in facets[key]:
                    facets[key][value] = facet_data
        return facets

    def process_group_attrs(self, state, parsed):
        result = {}
        if not parsed.has_grouping:
            return result

        for group in parsed.get_groups():
            result[group['category']['value']] = {'name': group['category']['attr']}
        return result

    def set_data(self, state, parsed):
        if not self.is_valid_response(state, parsed):
            return

        result = {
            'parsed': parsed,
        }
        state.searches[self.search['name']].update(result)


class FullSuggestSearchStep(ParallelStep):
    def get_steps(self, state=None):
        return [SimpleSearchStep(search) for search in state.searches.values()]


class OneSearchStep(SimpleSearchStep):
    def set_data(self, state, parsed):
        super().set_data(state, parsed)
        if state.errors:
            return

        try:
            result = {}
            query = parsed.request_query
            if query is not None:
                self.search['qtree'] = parse_query(query, 5)
            else:
                self.search['qtree'] = state.qtree

            if self.search['name'] == 'search_results':
                result['facets'] = self.process_facets(state, parsed)
            result['group_attrs'] = self.process_group_attrs(state, parsed)

            state.searches[self.search['name']].update(result)
        except Exception:
            log.exception('Cannot process search data: %s', self.search['name'],
                          extra={'state': state})
            self.set_error(state)

    def set_error(self, state, response=None):
        if self.search['name'] == 'search_results':
            state.set_error(ERROR_UNKNOWN)


class FullSearchStep(ParallelStep):
    def get_steps(self, state=None):
        if not state.searches:
            raise Exception('No searches found')

        return [OneSearchStep(search) for search in state.searches.values()]
