from logging import getLogger

from django.conf import settings

from intranet.search.abovemeta import errors
from intranet.search.abovemeta.request import Request
from intranet.search.abovemeta.steps.base import HttpStep


log = getLogger(__name__)


class BaseTVMStep(HttpStep):
    endpoint = settings.ISEARCH['api']['tvm2']
    name = None

    def get_request(self, state):
        headers = {'Authorization': settings.TVM2_AUTH_TOKEN}
        headers.update(self.get_headers(state))

        return Request(
            url=self.endpoint[self.name].url(query=self.get_query()),
            headers=headers,
            type_='tvm',
            name=self.name,
        )

    def get_headers(self, state):
        return {}

    def get_query(self):
        return {}

    def set_error(self, state, response):
        msg = response.body if response else 'Unknown error'
        code = response.code if response else None
        log.error('TVM %s failed: %s, %s', self.name, code, msg, extra={'state': state})
        state.set_error(errors.ERROR_AUTH, msg)


class CheckServiceTicketStep(BaseTVMStep):
    """ Стадия проверки tvm-тикета пришедшего к нам сервиса
    """
    name = 'check_service_ticket'

    def get_headers(self, state):
        return {settings.TVM2_SERVICE_HEADER: state.req_headers.get(settings.TVM2_SERVICE_HEADER)}

    def get_query(self):
        return {'dst': settings.TVM2_CLIENT_NAME}

    def set_data(self, state, auth_data):
        state.tvm_auth_service = auth_data['src']


class GetServiceTicketStep(BaseTVMStep):
    """ Стадия получения сервисного тикета для нашего приложения
    (для похода в blackbox и другие сервисы)
    """
    name = 'get_service_ticket'

    def get_query(self):
        query = {
            'src': settings.TVM2_CLIENT_NAME,
            'dsts': ','.join(settings.TVM2_DSTS),
        }
        return query

    def set_data(self, state, auth_data):
        tickets = {key: value['ticket'] for key, value in auth_data.items()}
        state.tvm_service_tickets = tickets
