import re

import json
from collections import defaultdict
from urllib.parse import quote_plus

from django import forms
from django.http import HttpResponseBadRequest
from django.shortcuts import get_object_or_404, redirect
from django.conf import settings

from intranet.search.core.models import Formula
from intranet.search.core.storages import FormulaStorage
from intranet.search.admin.template import render
from intranet.search.core.utils import http
from intranet.search.core import context


settings_context = context.SettingsContext()
searches = settings_context.searches_names()

services = {k: v['search']['query'].get('service', '') or k
                            for k, v in settings.ISEARCH['api']['saas'].items()}
services[''] = ''


def compile_formula(polynomial, compiled, service):
    # В продакшене можно добавить просто сконвертированную формулу
    path = services[service or 'intrasearch-wiki']  # для мета-формулы идем в intrasearch-wiki
    if not polynomial:
        return True, compiled

    polynomial = quote_plus(polynomial.replace('\n', '').replace(' ', ''))
    url = settings.ISEARCH['api']['converter'].url().format(formula=polynomial,
                                                            path=path)
    session = http.create_session()
    resp = session.get(url, timeout=5)

    if resp.ok:
        result = resp.content.strip()
        if not result:
            return False, {'error': 'Wrong formula syntax'}
        return True, result
    else:
        return False, resp.content


class FormulaForm(forms.Form):
    id = forms.IntegerField(required=False)
    search = forms.ChoiceField(choices=searches.items())
    index = forms.CharField(required=False)
    name = forms.CharField()
    service = forms.ChoiceField(choices=[(s, s) for s in services], required=False)
    polynomial = forms.CharField(required=False)
    compiled = forms.CharField(required=False)
    additional = forms.CharField(required=False)

    # Регулярка для чистки знаков формулы. Находит сочетания
    # последовательно идущих + и -, которые не понимает саас
    MATH_SIGN_CLEAN_RE = re.compile(r'\+([\r\n\s]*?)-')

    def clean_polynomial(self):
        polynomial = self.cleaned_data.get('polynomial')
        if not polynomial:
            return polynomial

        # заменяем последовательные +- на -, т.к. сочетание двух
        # знаков подряд саас больше не принимает
        polynomial = self.MATH_SIGN_CLEAN_RE.sub(r'-\1', polynomial)
        return polynomial

    def clean(self):
        id = self.cleaned_data.get('id')
        name = self.cleaned_data['name']
        search = self.cleaned_data['search']
        index = self.cleaned_data['index']
        service = self.cleaned_data.get('service', '')

        f = Formula.objects.filter(search=search, index=index, name=name, service=service)

        if id:
            f = f.exclude(id=id)

        if f.exists():
            self._errors['name'] = ['Not unique']

        return self.cleaned_data


class DeleteFormulaForm(forms.Form):
    id = forms.IntegerField()


def formulas(request):
    formulas_list = Formula.objects.order_by('name').all()
    result = defaultdict(lambda: defaultdict(list))
    for formula in formulas_list:
        result[formula.search][formula.index].append(formula)

    return render(request, 'isearch/formula/formula.html',
                  {'formulas': dict(result), 'searches': searches, 'services': services},
                  content_type='text/html')


def add_formula(request):
    f = FormulaForm(request.POST or None)

    if f.is_valid():
        polynomial = f.cleaned_data.get('polynomial')
        compiled = f.cleaned_data.pop('compiled', '')
        service = f.cleaned_data.get('service')
        success, result = compile_formula(polynomial, compiled, service)

        if not success:
            return HttpResponseBadRequest(json.dumps({'converter': result}),
                                          content_type='application/json')
        FormulaStorage().create(compiled=result, **f.cleaned_data)
        return redirect(formulas)
    else:
        return HttpResponseBadRequest(json.dumps({'form': f.errors}),
                                      content_type='application/json')


def delete_formula(request, id):
    formula = get_object_or_404(Formula, id=id)

    f = DeleteFormulaForm(request.POST or None)
    if f.is_valid():
        formula.delete()
        return redirect(formulas)
    else:
        return HttpResponseBadRequest(f.errors, content_type='text/html')
