from datetime import datetime, timedelta

from django.conf import settings
from django.core.paginator import EmptyPage, PageNotAnInteger, Paginator
from django.http import JsonResponse, HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.views.decorators.http import require_POST

from intranet.search.core import storages
from intranet.search.core.models import PushRecord
from intranet.search.core.storages import PushStorage
from intranet.search.core.swarm.pushes import restart_push
from intranet.search.admin.template import render
from .sources import get_organization


def dashbord(request):
    hours = 24
    organization = get_organization(request.GET)

    storage = storages.PushStorage()
    stats = storage.stats(since=datetime.now() - timedelta(hours=hours),
                          organization_id=organization.get('id'))

    return render(request, 'isearch/pushes/dashbord.html',
                  {'stats': stats, 'hours': hours, 'organization': organization},
                  content_type='text/html')


def push_log(request, search):
    organization = get_organization(request.GET)

    records = PushRecord.objects.filter(search=search).order_by('-start_time')
    if organization:
        records = records.filter(organization_id=organization['id'])
    if request.GET.get('status'):
        records = records.filter(status=request.GET['status'])

    paginator = Paginator(records, 100)

    page = request.GET.get('page')

    try:
        paged_records = paginator.page(page)
    except PageNotAnInteger:
        paged_records = paginator.page(1)
    except EmptyPage:
        paged_records = paginator.page(paginator.num_pages)

    return render(request, 'isearch/pushes/push_log.html',
                  {'search': search, 'organization': organization,
                   'paginator': paginator, 'records': paged_records})


def push_info(request, push_id):
    push = get_object_or_404(PushRecord, id=push_id)
    endpoint = settings.ISEARCH['api']['ydeploy']['logs-url']
    if settings.APP_NAME == 'isearch' and settings.YENV_TYPE == 'production':
        deploy_unit = 'celery-pushes'
    else:
        deploy_unit = 'celery'
    query = {
        'query': f'context.context.push_id={push_id}',
        'deploy-unit': deploy_unit,
    }
    data = {
        'push': push,
        'deploy_log_url': endpoint.url(query=query, safe=':'),
    }
    return render(request, 'isearch/pushes/push_info.html', data)


def push_meta_json(request, push_id):
    push = get_object_or_404(PushRecord, id=push_id)
    return JsonResponse(push.meta)


@require_POST
def push_restart(request, push_id):
    get_object_or_404(PushRecord, id=push_id)
    restart_push(push_id, only_failed=False,
                 comment=f'manual restart from admin by user: {request.yauser.login}')
    return HttpResponseRedirect(request.META.get('HTTP_REFERER'))


@require_POST
def push_mark_as_known(request, push_id):
    PushStorage().update(push_id, status=PushStorage.STATUS_KNOWN_FAIL)
    return HttpResponseRedirect(request.META.get('HTTP_REFERER'))
