import logging
from functools import partial

from django import forms

from intranet.search.admin.admin.views.fields import ScopeField
from intranet.search.core.utils import QueryDict
from intranet.search.core import context
from intranet.search.admin import template

from intranet.search.admin.utils import get_abovemeta_response, BaseSearchForm


settings_context = context.SettingsContext()

views = [('', '---')]
for scope in settings_context.scopes:
    v = settings_context.views_for_scope(scope)
    if v:
        for view in v:
            views.append((view[0], view[1]['name']['ru']))


log = logging.getLogger(__name__)


class SearchForm(BaseSearchForm):
    scope = ScopeField(label='Область')

    content = forms.ChoiceField(label='Контент', required=False,
                                choices=(('', ''),
                                         ('translated', 'translated'),
                                         ('raw', 'raw'),
                                         ('archive', 'archive'),
                                         ('external', 'external')))
    features = forms.CharField(label='Фичи', required=False,
                               widget=forms.Textarea(
                                   attrs={'rows': "2", 'placeholder': 'people_formula=test'}))
    wizard = forms.ChoiceField(label='View', required=False,
                               choices=views)
    show_all_factors = forms.BooleanField(label='Показать все факторы', required=False)
    sort_by_time = forms.BooleanField(label='Отсортировать по времени', required=False)

    def clean_features(self):
        features = self.cleaned_data['features'].split()
        try:
            result = dict(f.split('=', 1) for f in features)
        except ValueError:
            self._errors['features'] = ['Некорректный формат.']
            result = ''
        return result


class AbovemetaPaginator:

    def __init__(self, result, query, offset=5):
        """ Псевдо-пагинатор по json выдаче

        :param result: результат ответа abovemeta
        :param offset: количество страниц, которое нужно выводить вокруг текущей
        """
        self.offset = offset
        self.query = query
        self.count = result['meta']['count']
        self.page = result['meta']['page']
        self.per_page = result['meta']['per_page']
        self.num_pages = result['meta']['pages']

    @property
    def pages(self):
        last_page = self.page + self.offset + 1
        if last_page > self.num_pages:
            last_page = self.num_pages

        first_page = self.page - self.offset
        if first_page < 0:
            first_page = 0

        return range(first_page, last_page)

    @property
    def next_num(self):
        next = self.page + self.offset + 1
        if next > self.num_pages:
            return 0
        return next

    @property
    def next_url(self):
        return self.build_query(self.next_num)

    @property
    def previous_num(self):
        previous = self.page - self.offset - 1
        if previous > 0:
            return previous
        return 0

    @property
    def previous_url(self):
        return self.build_query(self.previous_num)

    def build_query(self, page):
        query_dict = QueryDict()
        query_dict.update(self.query)
        query_dict['p'] = page
        return query_dict.urlencode()


def build_facet_query(query, facet_name=None, facet_value=None):
    query_dict = QueryDict()
    query_dict.update(query)
    query_dict['p'] = 0
    if facet_name:
        query_dict['facet.%s' % facet_name] = facet_value
    else:
        for key in query_dict.keys():
            if key.startswith('facet.'):
                del query_dict[key]
    return query_dict.urlencode()


def search_viewer(request):
    form = SearchForm(request.GET or None)
    response_data = {'form': form}

    if form.is_valid():
        query = QueryDict()
        query.update({'format': 'json', 'debug': True})

        for name, value in form.cleaned_data['features'].items():
            query.update({f'feature.{name}': value})

        if form.cleaned_data['wizard']:
            query.update({'wizard.search_results': form.cleaned_data['wizard']})

        if form.cleaned_data['sort_by_time']:
            query.update({'sorted': 'date'})

        query.update(request.GET)
        params = {
            'cookies': request.COOKIES,
            'ip': request.META.get('HTTP_X_REAL_IP'),
        }
        if form.cleaned_data['user_uid']:
            params['uid'] = form.cleaned_data['user_uid']
        if form.cleaned_data['cloud_uid']:
            params['cloud_uid'] = form.cleaned_data['cloud_uid']

        response = get_abovemeta_response('abovemeta', query, params, form.cleaned_data['organization'])
        response_data['abovemeta_url'] = response.url

        try:
            response_data['result'] = response.json()
        except Exception:
            response_data['response_error'] = {'status_code': response.status_code,
                                               'content': response.content}
        else:
            response_data['paginator'] = AbovemetaPaginator(response_data['result'], request.GET)
            response_data['build_facet_query'] = partial(build_facet_query, request.GET)

    return template.render(request, 'isearch/search/search_viewer.html',
                           response_data,
                           content_type='text/html')
