import collections
import datetime
import logging
import os

from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError
from django.http import HttpResponse, HttpResponseBadRequest
from django.shortcuts import get_object_or_404, redirect
from django.views.decorators.http import require_POST

from intranet.search.core import storages
from intranet.search.core.models import Indexation
from intranet.search.core.storages.base import DoesNotExist
from intranet.search.core.swarm.indexer import Indexer, STORAGES
from intranet.search.core.swarm.tasks import reindex, pause_index, continue_index
from intranet.search.core.utils import convert_timeout, get_possible_suffixes, json
from intranet.search.core.utils.saas import get_doc_count, get_indextstat_doc_count
from intranet.search.admin.admin.utils import paginate
from intranet.search.admin import template

log = logging.getLogger(__name__)


class PausedIndexation(forms.Form):
    indexation_id = forms.IntegerField()
    comment = forms.CharField()


class RenewedIndexation(forms.Form):
    indexation_id = forms.IntegerField()


class StopIndexation(forms.Form):
    indexation_id = forms.IntegerField()
    status = forms.CharField()
    from_list = forms.BooleanField(required=False)


class StartIndexation(forms.Form):
    limit = forms.IntegerField(required=False, initial=Indexer.default_options['limit'])
    ts = forms.DateTimeField(required=False, initial=Indexer.default_options['ts'])
    debug = forms.BooleanField(required=False, initial=Indexer.default_options['debug'])
    dry_run = forms.BooleanField(required=False, initial=Indexer.default_options['dry_run'])
    shadow_revision = forms.BooleanField(required=False, initial=Indexer.default_options['shadow_revision'])
    threads = forms.IntegerField(required=False, initial=Indexer.default_options['threads'])
    nort = forms.BooleanField(required=False, initial=Indexer.default_options['nort'])
    global_ = forms.BooleanField(required=False, initial=Indexer.default_options['global'])
    flush = forms.BooleanField(required=False, initial=Indexer.default_options['flush'])

    search = forms.CharField()
    index = forms.CharField(required=False)
    use_ts = forms.BooleanField(required=False)
    new_revision = forms.BooleanField(required=False)
    revision_id = forms.IntegerField(required=False)
    service = forms.CharField(required=False)
    keys = forms.CharField(required=False)
    nolock = forms.BooleanField(required=False)
    lock_timeout = forms.CharField(required=False)
    comment = forms.CharField()
    priority = forms.IntegerField(required=False, initial=Indexer.default_options['priority'])
    suffix = forms.CharField(required=False)
    load_percent = forms.IntegerField()
    no_check = forms.BooleanField(required=False)
    organizations = forms.CharField(required=False)
    document_storages = forms.MultipleChoiceField(choices=list(zip(STORAGES, STORAGES)))
    from_cache = forms.BooleanField(required=False)
    cache_table = forms.CharField(required=False)

    def clean_index(self):
        return self.cleaned_data.get('index') or ''

    def clean_new_revision(self):
        return self.cleaned_data.get('new_revision') or bool(self.cleaned_data.get('revision_id'))

    def clean_lock_timeout(self):
        return self.cleaned_data.get('lock_timeout') or '1h'

    def clean_ts(self):
        ts = self.cleaned_data.get('ts')
        if ts is not None:
            ts = ts.strftime('%s')
        return ts

    def clean_threads(self):
        return self.cleaned_data.get('threads') or 0

    def clean_priority(self):
        priority = int(self.cleaned_data.get('priority', 0))
        if priority > 9:
            self._errors['priority'] = ['Value must be between 0 and 9']
        return priority

    def clean_host(self):
        return self.cleaned_data['host'] or None

    def clean_organizations(self):
        organizations = self.cleaned_data['organizations'].strip()
        if not organizations and settings.APP_NAME == 'bisearch':
            raise ValidationError('Organizations is required')
        return organizations

    def clean_document_storages(self):
        return ','.join(self.cleaned_data['document_storages'] or [])


class RepeatIndexation(forms.Form):
    comment = forms.CharField(required=False)
    original_id = forms.IntegerField()


class RevisionForm(forms.Form):
    revision_id = forms.IntegerField()
    search = forms.CharField()
    index = forms.CharField(required=False)
    on_air = forms.BooleanField(required=False)
    delete = forms.BooleanField(required=False)


class FailIndexationsForm(forms.Form):
    search = forms.CharField(required=False)
    index = forms.CharField(required=False)
    threshold = forms.CharField()

    def clean_threshold(self):
        threshold = self.cleaned_data.get('threshold')
        try:
            return convert_timeout(threshold)
        except Exception as e:
            self._errors['threshold'] = ['Error while converting threshold to seconds: %s' % e]
            return threshold


class CountForm(forms.Form):
    revision_id = forms.IntegerField()
    indexation_id = forms.IntegerField(required=False)
    use_indexstat = forms.BooleanField(required=False)


class OrganizationForm(forms.Form):
    organization_id = forms.IntegerField()


def determine_index(index):
    if index == 'default':
        return ''
    return index


def get_organization(get_request):
    form = OrganizationForm(get_request)
    org_repository = storages.OrganizationStorage()

    organization = {}
    if form.is_valid():
        organization = org_repository.get(form.cleaned_data['organization_id'])

    return organization


def sources(request):
    recent = {}
    repository = storages.StorageRepository()

    organization = get_organization(request.GET)
    if organization:
        params = {'organization_id': organization['id']}
    else:
        params = {'order_by': 'organization__label', 'limit': 5}

    for search_id, item in settings.ISEARCH['searches']['base'].items():
        recent[search_id] = {}

        for index_id, index in item['indexes'].items():
            recent[search_id][index_id] = []

            try:
                revisions = repository['revision'].get_active(search_id, index_id, 'platform', **params)
            except DoesNotExist:
                continue

            for revision in revisions:
                revision_id = revision['id']
                last = repository['indexation'].get_latest_indexation(
                    search=search_id,
                    index=index_id,
                    revision=revision_id,
                    status__in=['done', 'fail'],
                )
                current = repository['indexation'].get_latest_indexation(
                    search=search_id,
                    index=index_id,
                    revision=revision_id,
                    status__in=['new'],
                )
                result = {
                    'current': current,
                    'finished': last,
                    'revision': revision,
                }
                recent[search_id][index_id].append(result)

    data = {
        'isearch': settings.ISEARCH,
        'suffixes': get_possible_suffixes(),
        'recent_indexations': recent,
        'defaults': Indexer.default_options,
        'organization': organization,
    }

    return template.render(request, 'isearch/sources/index.html',
                           data, content_type='text/html')


def sources_indexation_list(request, search_id, index):
    index = determine_index(index)

    organization = get_organization(request.GET)
    params = {'revision__organization_id': organization['id']} if organization else {}

    logs = (Indexation.objects.filter(search=search_id, backend='platform', index=index, **params)
                              .select_related('revision')
                              .order_by('-id'))

    cur_logs = paginate(logs, request.GET.get('page'))
    errors = [l for l in cur_logs if l.status != 'done' or l.end_time is None]

    data = {
        'name': search_id,
        'logs': cur_logs,
        'errors': errors,
        'index': index,
        'backend': 'platform',
        'suffixes': get_possible_suffixes(),
        'defaults': Indexer.default_options,
        'organization': organization,
    }
    return template.render(request, 'isearch/sources/indexation-list.html',
                           data,
                           content_type='text/html')


def doc_count(request):
    f = CountForm(request.GET)
    if f.is_valid():
        revisions_storage = storages.RevisionStorage()
        revision = revisions_storage.get_by_id(f.cleaned_data['revision_id'])

        if f.cleaned_data['use_indexstat']:
            result = get_indextstat_doc_count(revision)
        else:
            indexation_id = f.cleaned_data['indexation_id']

            if indexation_id is not None:
                text = 'i_indexation_id:%s' % indexation_id
            else:
                text = 'url:"*"'
            result = get_doc_count(revision, text)

        return HttpResponse(json.dumps(result),
                            content_type='application/json')
    else:
        return HttpResponseBadRequest(f.errors, content_type='text/html')


def _form_stages_data(stages_data, stages):
    data = []
    for stage in stages:
        result = []
        for status in ('new', 'in progress', 'retry', 'done', 'fail', 'cancel'):
            result.append((status, stages_data.get(stage, {}).get(status, 0)))
        data.append((stage, result))
    return data


def process_stage_stats(*indexation_ids):
    STAGES = ('setup', 'walk', 'fetch', 'load', 'create', 'content', 'store')

    indexation_storage = storages.IndexationStorage()

    data = {}
    stage_stats = indexation_storage.get_combined_stage_stats(indexation_ids)
    if stage_stats.is_empty():
        data['stages'] = _form_stages_data(storages.StageStatusStorage.get_default_stats(), STAGES)
    else:
        res = collections.defaultdict(dict)
        for (stage, status), info in stage_stats:
            res[stage][status] = info['total']
        data['stages'] = _form_stages_data(res, STAGES)
        data['ticks'] = stage_stats.tick
        data['plot_data'] = stage_stats.to_json()
        data['hosts'] = '\n'.join(
            '%s: %s' % h for h in indexation_storage.get_combined_hosts(indexation_ids, with_date=True)
        )

    return data


def sources_indexation(request, search_id, index, indexation_id):
    indexation = get_object_or_404(Indexation, id=indexation_id)
    indexation_storage = storages.IndexationStorage()
    indexer = indexation_storage.get_indexer(indexation_id)
    options = indexer.options if indexer else {}

    data = {
        'indexation': indexation,
        'options': options,
    }

    data.update(process_stage_stats(indexation_id))

    if os.environ.get('DEPLOY_STAGE_ID'):
        endpoint = settings.ISEARCH['api']['ydeploy']['logs-url']
        query = {'query': f'context.context.indexation_id={indexation_id}', 'deploy-unit': 'celery'}
        data['deploy_log_url'] = endpoint.url(query=query, safe=':')
    else:
        endpoint = settings.ISEARCH['api']['qloud']['celery']
        query = {'tab': 'logs-beta', 'query': '# @fields.context.indexation_id:%s' % indexation_id}
        data['deploy_log_url'] = endpoint.url(query=query, safe=':')

    return template.render(request, 'isearch/sources/indexation.html', data,
                           content_type='text/html')


@require_POST
def sources_fail_indexations(request):
    f = FailIndexationsForm(request.POST)

    if f.is_valid():
        threshold = f.cleaned_data.get('threshold')
        search = f.cleaned_data.get('search')
        index = f.cleaned_data.get('index', '')

        threshold_date = datetime.datetime.now() - datetime.timedelta(seconds=threshold)
        indexations = Indexation.objects.filter(status__in=['new', 'stop'],
                                                start_time__lt=threshold_date)
        if search:
            indexations = indexations.filter(search=search, index=index)

        for indexation in indexations:
            try:
                indexer = storages.IndexationStorage().get_indexer(indexation.id)
            except:
                indexer = None

            if indexer:
                message = f'Bulk fail. Threshold: {threshold_date}. Author: {request.yauser.login}'
                indexer.finish('fail', message)
            else:
                storages.IndexationStorage().update(indexation.id, status='fail')

        destination = request.META.get('HTTP_REFERER') or sources_all_indexations
        return redirect(destination)
    else:
        return HttpResponseBadRequest(f.errors, content_type='text/html')


@require_POST
def sources_stop_indexation(request):
    f = StopIndexation(request.POST)

    if f.is_valid():
        indexation_id = f.cleaned_data['indexation_id']
        status = f.cleaned_data['status']

        indexer = storages.IndexationStorage().get_indexer(indexation_id)

        message = f'Status changed from admin to: {status}. Author: {request.yauser.login}'
        if indexer:
            indexer.options['noqueue'] = False
            if status == 'stop':
                indexer.stop(message)
            else:
                indexer.finish(status, message)
        else:
            storages.IndexationStorage().update(indexation_id, status=status,
                                                msg=message)

        destination = request.META.get('HTTP_REFERER') or sources
        return redirect(destination)
    else:
        return HttpResponseBadRequest(f.errors, content_type='text/html')


@require_POST
def sources_new_indexation(request):
    return _start_indexation(request.POST, user=request.yauser.login)


@require_POST
def sources_repeat_indexation(request):
    f = RepeatIndexation(request.POST or None)
    if f.is_valid():
        original_id = f.cleaned_data['original_id']
        comment = '{} [restart #{}]'.format(f.cleaned_data['comment'], original_id)

        istorage = storages.IndexationStorage()
        params = istorage.get(original_id)
        params.update(istorage.get_indexer(original_id).options)
        params['revision_id'] = params['revision']['id']
        params['organizations'] = params['revision']['organization']['id']
        params['keys'] = ', '.join(params['keys']) if params.get('keys') else None
        params['ts'] = datetime.datetime.fromtimestamp(int(params['ts'])) if params.get('ts') else None
        params['comment'] = comment
        params.pop('push_ids', None)

        return _start_indexation(params, user=request.yauser.login)
    else:
        return HttpResponseBadRequest(f.errors, content_type='text/html')


@require_POST
def sources_pause_indexation(request):
    f = PausedIndexation(request.POST or None)
    if f.is_valid():
        indexation_id = f.cleaned_data['indexation_id']
        comment = '{} [pause #{}]'.format(f.cleaned_data['comment'], indexation_id)

        istorage = storages.IndexationStorage()
        params = istorage.get(indexation_id)
        params.update({
            'indexation_id': indexation_id,
            'comment': comment,
        })
        return _pause_indexation(params, user=request.yauser.login)
    else:
        return HttpResponseBadRequest(f.errors, content_type='text/html')


@require_POST
def sources_continue_indexation(request):
    f = RenewedIndexation(request.POST or None)
    if f.is_valid():
        indexation_id = f.cleaned_data['indexation_id']
        return continue_index(indexation_id, user=request.yauser.login)
    else:
        return HttpResponseBadRequest(f.errors, content_type='text/html')


def _pause_indexation(params, user):
    f = PausedIndexation(params or None)
    if f.is_valid():
        params = f.cleaned_data
        indexation_id = params.pop('indexation_id')
        search = params.pop('search')
        index = params['index']

        params['user'] = user
        params['global'] = params.pop('global_', Indexer.default_options['global'])

        pause_index(indexation_id, **params)

        return redirect(sources_indexation_list, search, index or 'default')
    else:
        return HttpResponseBadRequest(json.dumps({'form': f.errors}),
                                      content_type='application/json')


def _start_indexation(params, user):
    f = StartIndexation(params or None)
    if f.is_valid():
        params = f.cleaned_data
        search = params.pop('search')
        index = params['index']

        params['user'] = user
        params['global'] = params.pop('global_', Indexer.default_options['global'])

        indexation_id = reindex(search, **params)

        if indexation_id:
            return redirect(sources_indexation, search, index or 'default', indexation_id)
        else:
            return redirect(sources_indexation_list, search, index or 'default')
    else:
        return HttpResponseBadRequest(json.dumps({'form': f.errors}),
                                      content_type='application/json')


def sources_revision_list(request, search_id, index):
    index = determine_index(index)

    organization = get_organization(request.GET)
    params = {'organization_id': organization['id']} if organization else {}

    revisions_storage = storages.RevisionStorage()
    revisions = collections.OrderedDict()
    for status in ['active', 'ready', 'new', 'broken', 'deleted']:
        revision = revisions_storage.get_by_status(
            search=search_id,
            index=index,
            backend='platform',
            status=status,
            limit=20,
            order_by='-date',
            **params
        )

        revisions[status] = revision

    data = {
        'revisions': revisions,
        'search_id': search_id,
        'index': index,
        'organization': organization,
    }

    return template.render(request, 'isearch/sources/revision-list.html', data,
                           content_type='text/html')


def sources_revision(request, search_id, index, revision_id):
    # TODO: doc count from ts
    indexation_storage = storages.IndexationStorage()
    revisions_storage = storages.RevisionStorage()
    revision = revisions_storage.get_by_id(revision_id)
    artefact_repository = storages.ArtefactStorageRepository(revision)

    stage_stats = indexation_storage.get_stage_stats(None, revision_id)
    res = collections.defaultdict(dict)
    for (stage, status), info in stage_stats:
        res[stage][status] = info['total']

    data = {
        'revision': revision,
        'group_attrs': artefact_repository['group_attr'].count(),
        'indexations': indexation_storage.get_last(revision=revision_id),
        'indexations_count': indexation_storage.count(revision=revision_id),
    }

    return template.render(request, 'isearch/sources/revision.html', data,
                           content_type='text/html')


@require_POST
def sources_manage_revision(request):
    f = RevisionForm(request.POST)

    if f.is_valid():
        revision_id = f.cleaned_data['revision_id']
        search = f.cleaned_data['search']
        index = f.cleaned_data['index']
        delete = f.cleaned_data['delete']
        on_air = f.cleaned_data['on_air']
        revisions_storage = storages.RevisionStorage()
        revision = revisions_storage.get_by_id(revision_id)
        if on_air:
            revisions_storage.set_active(revision)
        elif delete:
            revisions_storage.purge(revision_id)

        if request.META.get('HTTP_REFERER'):
            return redirect(request.META.get('HTTP_REFERER'))

        return redirect(sources_revision_list, search, index)
    else:
        return HttpResponseBadRequest(json.dumps({'form': f.errors}),
                                      content_type='application/json')


def sources_all_indexations(request):
    organization = get_organization(request.GET)
    params = {'revision__organization_id': organization['id']} if organization else {}

    indexations = (Indexation.objects.filter(backend='platform', **params)
                                     .select_related('revision', 'revision__organization')
                                     .order_by('-id'))
    cur_indexations = paginate(indexations, request.GET.get('page'))

    data = {
        'logs': cur_indexations,
        'defaults': Indexer.default_options,
        'suffixes': get_possible_suffixes(),
        'organization': organization,
    }

    return template.render(request, 'isearch/sources/indexation-list.html',
                           data,
                           content_type='text/html')
