from collections import defaultdict
from urllib.parse import parse_qsl

from django import forms
from django.conf import settings
from django.http import HttpResponseBadRequest, HttpResponse

import json

from intranet.search.admin import template

from intranet.search.admin.utils import get_abovemeta_response, BaseSearchForm


LAYERS_CHOICES = [(k, k) for k in sorted(settings.ISEARCH['suggest']['layers']) if k != 'base']


class SuggestForm(BaseSearchForm):
    version = forms.ChoiceField(label='Версия', choices=(('0', '0'), ('1', '1'), ('2', '2')))
    layers = forms.MultipleChoiceField(label='Слои', choices=LAYERS_CHOICES, required=False)
    show_timings = forms.BooleanField(label='Показать тайминги', required=False)
    extra_query = forms.CharField(label='Дополнительные GET-параметры', required=False,
                                  widget=forms.Textarea(attrs={'rows': 3}))

    def clean_extra_query(self):
        extra_query = self.cleaned_data.get('extra_query', '')
        if not extra_query:
            return {}
        return dict(parse_qsl(extra_query, keep_blank_values=True))

    def clean_layers(self):
        return ','.join(self.cleaned_data.get('layers', []))


def plot_data(suggest_requests):
    if not suggest_requests:
        return
    data = defaultdict(list)
    labels = []

    for req in suggest_requests:
        name = '{}_{}'.format(req['type'], req['name'])
        if name not in labels:
            labels.append(name)

        ind = labels.index(name)
        data[name] += [[req['started'], ind], [req['started']+req['duration'], ind], [None, ind]]

    values = []
    for label in labels:
        values.append(data[label])

    labels = list(enumerate(labels))

    return {'labels': labels, 'values': values}


def suggest(request):
    form = SuggestForm(request.GET or None)

    if form.is_valid():
        query = form.cleaned_data
        query['user'] = request.yauser.login
        if int(query.get('version', 0)) == 0:
            query['s[]'] = ['nav', 'people']
        query['feature.full_suggest_info'] = True
        query.update(query.pop('extra_query', {}))

        params = {
            'cookies': request.COOKIES,
            'ip': request.META.get('HTTP_X_REAL_IP'),
        }
        if form.cleaned_data['user_uid']:
            params['uid'] = form.cleaned_data['user_uid']
        if form.cleaned_data['cloud_uid']:
            params['cloud_uid'] = form.cleaned_data['cloud_uid']

        response = get_abovemeta_response('suggest', query, params, form.cleaned_data['organization'])

        if response.ok:
            response_json = response.json()

            suggest_requests = None
            version = int(query.get('version', 0))
            if version == 0:
                suggest_requests = response_json.pop('requests', None)
            elif version == 1:
                for i, data in enumerate(response_json):
                    if data['layer'] == 'full_suggest_info':
                        response_json.pop(i)
                        suggest_requests = data['result']['requests']
                        break
            elif version == 2:
                suggest_requests = response_json.pop('full_suggest_info')['requests']

            data = {
                'error': False,
                'result': response_json,
                'requests_info': suggest_requests,
            }
            if query.get('show_timings'):
                data['requests'] = plot_data(suggest_requests)
        else:
            data = {
                'result': {
                    'code': response.status_code,
                    'reason': response.reason,
                },
                'error': True
            }

        return HttpResponse(json.dumps(data), content_type='application/json')
    else:
        return HttpResponseBadRequest(
            json.dumps({'error': True, 'result': form.errors}),
            content_type='application/json')


def suggest_viewer(request):
    form = SuggestForm(initial=(request.GET or {}))

    return template.render(request, 'isearch/search/suggest_viewer.html',
                           {'form': form},
                           content_type='text/html')
