import json
from logging import getLogger

from django import http
from django.core.exceptions import ObjectDoesNotExist
from django.core.serializers.json import DjangoJSONEncoder
from django.utils.encoding import force_text
from django.views.generic import View
from ylog.context import log_context

from ..forms import Form

logger = getLogger(__name__)


class BaseApiView(View):
    """ Базовая вьюха апи поиска
    """
    def dispatch(self, request, *args, **kwargs):
        try:
            return super().dispatch(request, *args, **kwargs)
        except Exception as e:
            return self.handle_exception(e)

    def handle_exception(self, exception):
        raise exception


class FormApiView(BaseApiView):
    """ Базовая вьюха апи поиска, использующих форму для валидации запроса
    """
    form_class = Form

    def get(self, request, *args, **kwargs):
        return self.process_form(request.GET)

    def post(self, request, *args, **kwargs):
        with log_context(user=request.yauser.login):
            logger.info('FormApiView user')
        content_type = request.META.get('CONTENT_TYPE', '')
        content_type_wo_charset = content_type.split(';', 1)[0]
        if content_type_wo_charset in ('application/json', 'text/plain'):
            try:
                return self.process_form(json=json.loads(request.body), user=request.yauser)
            except ValueError:
                return http.HttpResponseBadRequest('No JSON object could be decoded',
                                                   content_type='text/plain')
        else:
            return self.process_form(request.POST, user=request.yauser)

    def get_form(self, *args, **kwargs):
        return self.form_class(*args, **kwargs)

    def get_response_data(self, form):
        raise NotImplementedError

    def process_form(self, *args, **kwargs):
        form = self.get_form(*args, **kwargs)
        if form.validate():
            response_data = self.get_response_data(form)
            return http.HttpResponse(json.dumps(response_data, cls=DjangoJSONEncoder), content_type='application/json')
        else:
            errors = {
                key: [force_text(item) for item in value]
                for key, value in form.errors.items()
            }
            logger.info('Form is invalid: %s', errors)
            return http.HttpResponseBadRequest(json.dumps(errors), content_type='text/plain')

    def handle_exception(self, exception):
        if isinstance(exception, ObjectDoesNotExist):
            return http.HttpResponseNotFound('object not found', content_type='text/plain')
        return super().handle_exception(exception)


class ModernFormApiView(FormApiView):
    def process_form(self, *args, **kwargs):
        form = self.get_form(*args, **kwargs)
        if form.validate():
            response_data = self.get_response_data(form)
            return http.HttpResponse(json.dumps(response_data), content_type='application/json')
        else:
            errors = form.errors.copy()
            errors['__all__'] = []
            response_data = {
                'errors': errors,
            }
            serialized = json.dumps(response_data, cls=DjangoJSONEncoder)
            logger.info('Form is invalid: %s', form.errors)
            return http.HttpResponseBadRequest(serialized, content_type='application/json')
