from logging import getLogger
from functools import partial
from collections import defaultdict

from django.conf import settings
from django.db.transaction import atomic

from intranet.search.core import storages
from intranet.search.api.api1.views.base import FormApiView
from intranet.search.api.api1.forms import ApiDirectoryEventForm, ApiDirectorySyncForm
from intranet.search.core.tasks import (
    sync_directory_organization, sync_directory_object, sync_department_group_members
)
from intranet.search.core.swarm.pushes import log_push, kill_push
from intranet.search.core.sources.directory.client import (
    ORGANIZATION_EVENTS, USER_EVENTS, DOMAIN_EVENTS, SERVICE_EVENTS,
    DEPARTMENT_EVENTS, GROUP_EVENTS, GROUP_PROPERTY_CHANGED_EVENT, DEPARTMENT_PROPERTY_CHANGED_EVENT,
    USER_DELETED_EVENT
)

logger = getLogger(__name__)


AVAILABLE_SERVICES = list(settings.ISEARCH['searches']['base'])


def directory_service_event_handler(organization, **kwargs):
    """ Обработчик события подключения/выключения сервиса директории
    """
    service_slug = kwargs['object']['slug']
    if service_slug in AVAILABLE_SERVICES:
        sync_directory_organization.delay(organization, **kwargs)
    else:
        logger.info('Got event about unknown service %s. Skip it.', service_slug)
        kill_push(kwargs.get('push_id'), cancel=True, comment='Unknown service')


# словарь соответствия события директории обработчику
EVENT_HANDLER_MAP = defaultdict(list)
for event in ORGANIZATION_EVENTS:
    EVENT_HANDLER_MAP[event].append(sync_directory_organization.delay)
for event in DOMAIN_EVENTS:
    EVENT_HANDLER_MAP[event].append(partial(sync_directory_organization.delay, force_reindex=True))
for event in SERVICE_EVENTS:
    EVENT_HANDLER_MAP[event].append(directory_service_event_handler)
for event in USER_EVENTS:
    EVENT_HANDLER_MAP[event].append(partial(sync_directory_object.delay, index=''))
EVENT_HANDLER_MAP[USER_DELETED_EVENT].append(partial(sync_directory_object.delay, index='', action='delete'))

for event in DEPARTMENT_EVENTS:
    EVENT_HANDLER_MAP[event].append(partial(sync_directory_object.delay, index='departments'))
for event in GROUP_EVENTS:
    EVENT_HANDLER_MAP[event].append(partial(sync_directory_object.delay, index='groups'))

# при изменении свойств групп или департаментов дополнительно переиндексируем их членов
EVENT_HANDLER_MAP[GROUP_PROPERTY_CHANGED_EVENT].append(
    partial(sync_department_group_members.delay, type='group'))
EVENT_HANDLER_MAP[DEPARTMENT_PROPERTY_CHANGED_EVENT].append(
    partial(sync_department_group_members.delay, type='department'))

EVENT_INDEX_MAP = {}
for event in USER_EVENTS:
    EVENT_INDEX_MAP[event] = ''
for event in DEPARTMENT_EVENTS:
    EVENT_INDEX_MAP[event] = 'departments'
for event in GROUP_EVENTS:
    EVENT_INDEX_MAP[event] = 'groups'


class DirectoryEventHandlerApiView(FormApiView):
    """ Прием пушей от директории
    """
    form_class = ApiDirectoryEventForm

    def get_form(self, *args, **kwargs):
        query = kwargs.get('json')
        return self.form_class(**query)

    def post(self, request, *args, **kwargs):
        logger.info('Got request from Directory: body="%s"', request.body)
        return super().post(request, *args, **kwargs)

    @atomic
    def get_response_data(self, form):
        event = form.event.data
        event_handlers = EVENT_HANDLER_MAP.get(event)
        if not event_handlers:
            return {'status': 'error', 'details': 'Unknown event "%s"' % event}
        if event in SERVICE_EVENTS:
            service_object = form.data.get('object') or {}
            service_slug = service_object.get('slug', '')
            if service_slug not in AVAILABLE_SERVICES:
                logger.info('Got event about unknown service %s. Skip it.', service_slug)
                return {'status': 'ok', 'details': 'Unknown service "%s"' % service_slug, 'push_id': None}

        org_storage = storages.OrganizationStorage()
        org, is_new = org_storage.get_or_create(directory_id=form.org_id.data,
                                                defaults={'directory_revision': form.data.get('revision', 0)})
        push_id = log_push(form.object.data, 'directory', index=EVENT_INDEX_MAP.get(event, ''),
                           push_type=event, organization_id=org['id'])

        if is_new:
            # В случае создания новой организации явно переиндексируем её полностью
            sync_directory_organization.delay(org, force_reindex=True, push_id=push_id, event=event)
        else:
            for handler in event_handlers:
                handler(organization=org, push_id=push_id, **form.data)
        return {'status': 'ok', 'push_id': push_id}


class DirectorySyncOrganizationApiView(DirectoryEventHandlerApiView):
    """ Ручка принудительной синхронизации организации
    """
    form_class = ApiDirectorySyncForm

    @atomic
    def get_response_data(self, form):
        event = 'force_sync'
        org_storage = storages.OrganizationStorage()
        org, is_new = org_storage.get_or_create(directory_id=form.org_id.data)
        push_id = log_push({}, 'directory', push_type=event, organization_id=org['id'])
        sync_directory_organization.delay(org, force_reindex=True, push_id=push_id, event=event)
        return {'status': 'ok', 'push_id': push_id}
