import json
from datetime import timedelta, datetime

from django.conf import settings
from django.http import HttpResponse
from django.utils.timezone import utc

from intranet.search.core import sources
from intranet.search.core.models import Indexation, PushRecord
from intranet.search.core.query import ast
from intranet.search.core.sources.utils import date_to_timestamp
from intranet.search.core.storages import RevisionStorage, PushStorage
from intranet.search.core.utils import convert_timeout
from intranet.search.core.utils.saas import get_doc_count


def get_source_count(revision, updated_since, updated_till):
    """ Возвращает количество документов в источнике, измененых с времени updated
    """
    Source = sources.load(revision['search'], revision['index'])
    indexer = Source({'revision': revision})
    return indexer.objects_count(updated_since=updated_since, updated_till=updated_till)


def get_index_count(revision, updated_since, updated_till):
    """ Возвращает количество документов в нашем индексе, измененых с времени updated
    """
    text = (ast.AttrSearch('url', ast.OrderedText('*')) &
            ast.AttrGreaterEqualSearch('i_updated', updated_since) &
            ast.AttrLessEqualSearch('i_updated', updated_till))
    result = get_doc_count(revision, text.to_string())
    return result['count']


def index_fullness_check(request, search, index=''):
    """ Вьюха проверяющая полноту индекса.
    Сравнивает количество документов в индексе с количеством документов в источнике
    за определенную дельту
    """
    index = index or ''
    delta_start = convert_timeout(request.GET.get('delta_start') or '2m')
    delta = convert_timeout(request.GET.get('delta') or '1d')
    now = datetime.utcnow().replace(tzinfo=utc, microsecond=0)

    updated_till_dt = now - timedelta(seconds=delta_start)
    updated_till = date_to_timestamp(updated_till_dt)
    updated_since_dt = updated_till_dt - timedelta(seconds=delta)
    updated_since = date_to_timestamp(updated_since_dt)

    max_diff_percent = float(request.GET.get('diff_percent') or 0)

    try:
        revision = RevisionStorage().get_active(search, index, limit=1)[0]
        source_count = get_source_count(revision, updated_since_dt, updated_till_dt)
        index_count = get_index_count(revision, updated_since, updated_till)
        pushes = PushStorage().get_by_search_count(search=search, index=index,
                                                   date__gte=updated_since_dt,
                                                   date__lte=updated_till_dt)
        pushes_count = pushes.get((search, index), 0)
    except Exception as e:
        status_code = 500
        data = {'error': str(e)}
    else:
        diff = abs(source_count - index_count)

        try:
            diff_percent = diff * 100.0 / max(index_count, source_count)
        except ZeroDivisionError:
            diff_percent = 0

        status_code = 500 if diff_percent > max_diff_percent else 200
        data = {'search': search, 'index': index,
                'updated_since': updated_since_dt.isoformat(), 'updated_since_ts': updated_since,
                'updated_till': updated_till_dt.isoformat(), 'updated_till_ts': updated_till,
                'diff_percent': diff_percent, 'source_count': source_count,
                'index_count': index_count, 'pushes_count': pushes_count}

    return HttpResponse(json.dumps(data), content_type='application/json', status=status_code)


def indexation_failes_check(request):
    """ Проверка упавших индексаций

    Падает, если у какого-то источника упала delta последних индексаций подряд
    """
    delta = int(request.GET.get('delta') or 3)
    all_failed = []

    for search, item in settings.ISEARCH['searches']['base'].items():
        for index in item['indexes']:
            statuses = list(Indexation.objects
                            .filter(search=search, index=index)
                            .exclude(status='new')
                            .values_list('status', flat=True)
                            .order_by('-id')[:delta])
            if len(statuses) == delta and all(st == 'fail' for st in statuses):
                all_failed.append((search, index))

    status_code = 500 if all_failed else 200
    return HttpResponse(json.dumps(all_failed), content_type='application/json', status=status_code)


def pushes_failes_check(request):
    """ Мониторинг упавших пушей

    Загорается, если есть упавшие пуши, которые уже нельзя рестартить
    """
    max_retries = 3
    failed_pushes = (
        PushRecord.objects
        .filter(status=PushRecord.STATUS_FAIL, retries__gte=max_retries)
        .values_list('id', flat=True)
    )
    failed_count = failed_pushes.count()
    data = {'failed_count': failed_count, 'failed_pushes': []}
    if failed_count > 0:
        status_code = 500
        data['failed_pushes'] = list(failed_pushes[:100])
    else:
        status_code = 200
    return HttpResponse(json.dumps(data), content_type='application/json', status=status_code)
