import json
import itertools
from datetime import timedelta
from logging import getLogger

import requests
from django.conf import settings
from django.utils import timezone
from django import http

from intranet.search.core import storages
from intranet.search.api.monitorings.views.redis import get_mem_usage as get_redis_mem_usage

log = getLogger(__name__)

# Подробнее про суффиксы: https://wiki.yandex-team.ru/golovan/userdocs/aggregation-types/
# На агрегацию по группе/метагруппе (2 буквы посередине) нам всё равно, потому что мониторинг
# запущен на одном инстансе, плюс даже если это изменится, данные берутся из одной базы, поэтому
# как минимум суммировать здесь некорректно, корректнее брать минимум или максимум.

# Для асболютных значений (a) при агрегации по времени просто берём максимум (последний из x).
# Например, сигнал "количество новых индексаций": значение нужно асболютное,
# суммировать при агрегации некорректно.
COUNT_SYGOPT = 'axxx'

# Для дельт (d) при агрегации по времени берём сумму (m).
# Например, сигнал "кол-во упавших пушей": нужна дельта, чтобы понимать сколько упало за последнее
# время, при агрегации логично суммировать, чтобы также понять, сколько упало за час, сутки, etc.
DELTA_SYGOPT = 'dxxm'

# Для агрегации по максимальному значению берем максимальное значение (x).
MAX_SYGOPT = 'max'

indexation_storage = storages.IndexationStorage()
push_storage = storages.PushStorage()
organization_storage = storages.OrganizationStorage()


def make_signal(name, value, sygopt):
    return [f'{name}_{sygopt}', float(value)]


def get_indexation_status_signals(index_status_count, status, sygopt):
    signals = []
    for (search, index), signal_value in index_status_count[status].items():
        signal_name = f'{status}_{search}_{index}_indexations'
        signal = make_signal(signal_name, signal_value, sygopt)
        signals.append(signal)

    return signals


def get_status_indexations_signals():
    index_status_count = indexation_storage.get_by_status_count(status='new')
    return itertools.chain(
        get_indexation_status_signals(index_status_count, 'new', COUNT_SYGOPT),
    )


def get_push_signals():
    result = []
    push_data = push_storage.get_by_search_count(date__gte=timezone.now() - timedelta(hours=1))
    for (search, index), count in push_data.items():
        result.append(make_signal(f'pushes_{search}_{index}', count, DELTA_SYGOPT))
    result.append(make_signal('pushes', sum(push_data.values()), DELTA_SYGOPT))
    return result


def get_push_status_signals():
    result = []
    push_data = push_storage.get_by_search_count(add_status=True, date__gte=timezone.now() - timedelta(hours=1))
    for (search, index, status), count in push_data.items():
        status_name = push_storage.STATUSES[status]
        result.append(make_signal(f'pushes_{search}_{index}_{status_name}', count, DELTA_SYGOPT))
    return result


def get_organization_signals():
    signal_name = 'organizations'
    signal_value = organization_storage.count()
    return [
        make_signal(signal_name, signal_value, COUNT_SYGOPT),
        make_signal(signal_name, signal_value, DELTA_SYGOPT),
    ]


def get_saas_push_signals():
    url = settings.ISEARCH['api']['saas_logbroker']['stat'].url()
    response = requests.get(url, timeout=0.5)
    if not response.ok:
        log.warning('Cannot get saas push stat data: %s, %s',
                    response.status_code, response.content)
        return []
    return response.json()


def get_redis_mem_signals():
    try:
        mem_usage = get_redis_mem_usage()
    except Exception as e:
        log.exception(str(e))
        return []

    result = []
    for instance, usage in mem_usage.items():
        result.append(make_signal(f'redis_{instance}_mem_usage', usage, MAX_SYGOPT))
    return result


def unistat():
    return list(itertools.chain(
        get_status_indexations_signals(),
        get_push_signals(),
        get_push_status_signals(),
        get_organization_signals(),
    ))


def unistat_view(request):
    """
    Статистика состояния системы в целом, не зависящая от хостов
    """
    data = unistat()
    return http.HttpResponse(json.dumps(data))


def unistat_celery_view(request):
    """
    Статистика состояния хостов celery
    """
    data = list(itertools.chain(
        get_saas_push_signals(),
        get_redis_mem_signals(),
    ))
    return http.HttpResponse(json.dumps(data))
