import logging
import threading
from contextlib import contextmanager
import warnings

from django import db

from celery import Celery, signals
from celery.app import app_or_default


# Глобальное прилоение с MongoDB в облаке
global_app = Celery('global')
global_app.config_from_envvar('CELERY_CONFIG_MODULE')

# Локальное приложение с Redis на localhost
local_app = Celery('local')
local_app.config_from_envvar('CELERY_LOCAL_CONFIG_MODULE')

global_app.set_default()
global_app.set_current()


apps = {
    'global': global_app,
    'local': local_app,
}

app_lock = threading.RLock()


@contextmanager
def switch_app(app, *tasks):
    """Переключаем дефолтное приложение"""
    if isinstance(app, str):
        app = apps[app]

    with app_lock:
        old_app = app_or_default()
        app.set_current()

        for task in tasks:
            task.bind(app)

        try:
            yield app
        finally:
            old_app.set_current()

            for task in tasks:
                task.bind(old_app)


@signals.setup_logging.connect
def setup_logging(logfile, loglevel, **kwargs):
    # Наличие этого хендлера сигнала отключает настройку логирования в самом Celery.
    # Тут мы просто создаем лог-хендлер с prefix на базе имени файла
    root = logging.getLogger()

    if loglevel:
        root.setLevel(loglevel)

# Подчищаем протухшие соединения с базой после каждой таски
signals.task_prerun.connect(db.close_old_connections)

warnings.filterwarnings('ignore', category=DeprecationWarning, module='kombu.transport.mongodb')
