from copy import deepcopy
import weakref
from itertools import groupby
from urllib.parse import unquote, parse_qsl

from lxml import etree

from django.conf import settings

from kombu.utils import cached_property

from intranet.search.core.query import parse_query
from intranet.search.core.utils import get_kps


class BaseContext:
    def __init__(self, storage=None):
        self.storage = storage

    @property
    def search_context(self):
        try:
            return self._search_context
        except AttributeError:
            raise AttributeError('Search context is unapplied')

    def apply_search_context(self, search_context):
        self._search_context = weakref.proxy(search_context)


class SettingsContext(BaseContext):
    '''
    Содержит информацию о настройках системы
    '''
    def __init__(self):
        super().__init__()

        self.searches = dict(settings.ISEARCH.get('searches', {}).get('meta', {}),
                             **settings.ISEARCH.get('searches', {}).get('base', {}))

        self.scopes = settings.ISEARCH['scopes']

    def base_collection(self, search, index):
        try:
            return self.searches[search]['collections'][index], index
        except KeyError:
            if index == '':
                raise

            return self.searches[search]['collections'][''], ''

    def external_collection(self, search, index):
        return settings.ISEARCH['searches']['external'][search]['collections'][index], index

    def sources_for_meta(self, index):
        # Возвращает [(имя источника, индекс), ...] для метапоиска на платформе
        meta_index = self.searches['meta']['collections'][index]
        if not meta_index.get('use_meta_factors', False):
            return []

        return [(source, value) for (source, settings) in meta_index['sources'].items()
                                            for value in settings['collection']]

    def factors_for_search(self, search):
        factors = {'zone': [], 'static': []}
        if search in self.searches:
            factors.update(self.searches[search].get('factors', {}))
        return factors

    def meta_factors_for_search(self, search, index):
        is_in_meta = False
        for meta_index in self.searches.get('meta', {}).get('collections', []):
            if (search, index) in self.sources_for_meta(meta_index):
                is_in_meta = True
                break

        if is_in_meta:
            return self.factors_for_search('meta')

    def searches_names(self):
        return {search: values['name'] for search, values in self.searches.items()}

    @property
    def scope_has_facets(self):
        return bool(self.scopes[self.search_context.query.scope].get('facets'))

    def results_counters(self):
        return self.scopes[self.search_context.query.scope].get('results_count')

    def pages_counter(self):
        return self.scopes[self.search_context.query.scope].get('pages_count')

    def facets_for_scope(self):
        if not self.scope_has_facets:
            return

        scope = self.search_context.query.scope
        lang = self.search_context.query.language

        facets = sorted(self.scopes[scope]['facets'].items(),
                        key=lambda k: k[1].get('order', 100))

        return [(f[0], {'name': f[1].get('name', {}).get(lang, ''), 'order': f[1].get('order', '')})
                        for f in facets]

    def views_for_scope(self, scope):
        views = self.scopes[scope].get('views')
        if views:
            return sorted(views.items(), key=lambda k: k[1]['order'])

    @property
    def scope_has_sorts(self):
        return bool(self.scopes[self.search_context.query.scope].get('sort'))

    def sorts_for_scope(self):
        if self.scope_has_sorts:
            sort = sorted(self.scopes[self.search_context.query.scope]['sort'].items(),
                          key=lambda k: k[1].get('order', 100))
            return sort

    def show_facet(self, scope, facet):
        return self.scopes[scope]['facets'][facet].get('show_facet', True)


class QueryContext(BaseContext):
    '''
    Содержит информацию о параметрах поискового запроса
    '''
    def __init__(self, text=None, scope=None, content=None, language=None,
                 backend=None, facets=None, user=None, features=None,
                 storage=None, wizards=None, groups=None, wiki_cluster=None):
        super().__init__(storage)

        self.text = text
        self.scope = scope
        self.content = content
        self.language = language
        self.backend = backend
        self.facets = facets or {}
        self.user = user
        self.features = features or {}
        self.wizards = wizards or {}
        self.groups = groups or []
        self.wiki_cluster = unquote(wiki_cluster) if wiki_cluster else None
        self._qtree = None

    def __str__(self):
        facets = ','.join(self.facets)
        return (
            f'<Query text={self.text} scope={self.scope} content={self.content} '
            f'language={self.language} backend={self.backend} facets=[{facets}] '
            f'user={self.user} features={self.features}>'
        )

    @property
    def qtree(self):
        if self._qtree is None:
            self._qtree = parse_query(self.text, 5)

        return self._qtree

    @classmethod
    def from_kwargs(cls, **kwargs):
        return cls(**kwargs)

    def get_mask_field(self, attr, field_name):
        return {f'{field_name}.{name}': value
                                for name, value in getattr(self, attr, {}).items()}

    def get_features(self):
        return self.get_mask_field('features', 'feature')

    def get_wizards(self):
        return self.get_mask_field('wizards', 'wizard')

    def get_facets(self):
        return self.get_mask_field('facets', 'facet')


class UserContext(BaseContext):
    '''
    Содержит информацию о пользователе
    '''
    def __init__(self, user=None, groups=None, queues=None, storage=None, ip=None,
                 uid=None):
        super().__init__(storage)

        self.user = user
        self.groups = groups or []
        # TODO: В рамках USER_st_queue - набросок
        self.queues = queues or []
        self.ip = ip
        self.uid = uid

    def __str__(self):
        return '<User {} at groups: {}>'.format(self.user, ', '.join(map(str, self.groups)))

    @classmethod
    def from_raw(cls, user, groups, queues=None, ip=None, uid=None):
        # TODO: В рамках USER_st_queue - набросок (+queues)
        return cls(user, groups, queues, ip=ip, uid=uid)


class FeaturesContext(BaseContext):
    '''
    Содержит информацию об актуальных фичах
    '''
    def __init__(self, features=None, storage=None):
        super().__init__(storage)

        self.features = features or {}

    def __str__(self):
        return '<Features: {}>'.format(', '.join(f'{k}={v}' for k, v in self.features.items()))

    @classmethod
    def from_storage(cls, repository, user, groups):
        storage = repository['feature']

        features_user = {f['name']: f['value'] for f in storage.get(user=user)}
        features_group = {f['name']: f['value'] for f in storage.get(groups=groups)}

        features_group.update(features_user)

        return cls(features_group, storage=storage)

    @classmethod
    def from_raw(cls, data, **kwargs):
        return cls(features=data, **kwargs)

    def enabled(self, name):
        value = self.search_context.query.features.get(name, self.features.get(name))

        return value not in (None, '0', 'no', 'off')

    def value(self, name):
        return self.search_context.query.features.get(name, self.features.get(name))

    def to_raw(self):
        return self.features

    def to_xml(self):
        root = etree.Element('features', type='dict')
        features = dict(self.features.items() + self.search_context.query.features.items())
        for key, value in features.items():
            node = etree.Element(key)
            node.text = value
            root.append(node)
        return root


class RevisionsContext(BaseContext):
    '''
    Содержит информацию об актуальных ревизиях
    '''
    def __init__(self, revisions=None, storage=None):
        super().__init__(storage)

        self._revisions = revisions or {}

    def __str__(self):
        return '<Revisions: {}>'.format(', '.join('{} -> {}:{}'.format(k, v['service'], v['id'])
                                                        for k, v in self.revisions.items()))

    def __iter__(self):
        return self.revisions.values()

    @cached_property
    def revisions(self):
        defaults = self._revisions.copy()
        defaults.update(self.overloaded)

        return defaults

    @cached_property
    def overloaded(self):
        # TODO: это наверняка сломается
        def _gen():
            try:
                revisions = self.search_context.features.value('revisions')
            except AttributeError:
                return

            if revisions and self.storage:
                revisions = self.storage.get_by_id_list(revisions.split(','))

                for revision in revisions:
                    yield (revision['search'], revision['index'], revision['backend']), revision

        return dict(_gen())

    def find(self, search, index, backend='platform'):
        return self.revisions[(search, index or '', backend)]

    @classmethod
    def from_storage(cls, repository, organization_id):
        storage = repository['revision']
        return cls.from_raw(storage.get_all_active(organization_id=organization_id), storage=storage)

    @classmethod
    def from_raw(cls, data, **kwargs):
        revisions = {(r['search'], r['index'] or '', r['backend']): r
                                for r in data}

        return cls(revisions, **kwargs)

    def to_raw(self):
        return list(dict(search=k[0], index=k[1], backend=k[2], id=v['id'], service=v['service'])
                            for k, v in self.revisions.items())


class SearchContext:
    '''
    Содержит информацию о всем поисковом запросе
    '''
    def __init__(self, query=None, user=None, features=None,
                 revisions=None, settings=None, formulas=None, external_wizard_rules=None):
        # TODO: не нужна ли тут тоже организация?
        self.replace(query=query or QueryContext(),
                     user=user or UserContext(),
                     features=features or FeaturesContext(),
                     revisions=revisions or RevisionsContext(),
                     settings=settings or SettingsContext(),
                     formulas=formulas or FormulaContext(),
                     external_wizard_rules=external_wizard_rules or ExternalWizardRuleContext())

    def __deepcopy__(self, memo):
        context = type(self)()

        for name, nested in self.__dict__.items():
            new_nested = deepcopy(nested, memo)

            try:
                context.apply_search_context(context)
            except AttributeError:
                pass

            setattr(context, name, new_nested)

        return context

    @classmethod
    def from_storage(cls, repository, user, **kwargs):
        features = FeaturesContext.from_storage(repository, user, groups=kwargs.pop('groups', []))
        revisions = RevisionsContext.from_storage(repository,
                                                  organization_id=kwargs.pop('organization', None))
        formulas = FormulaContext.from_storage(repository)
        external_wizard_rules = ExternalWizardRuleContext.from_storage(repository)

        return cls(features=features, revisions=revisions, formulas=formulas,
                   external_wizard_rules=external_wizard_rules, **kwargs)

    def replace(self, **kwargs):
        for name, context in kwargs.items():
            context = deepcopy(context)
            context.apply_search_context(self)

            setattr(self, name, context)

    def parse_target(self, target):
        bits = target.split(':')

        try:
            target, collection = bits
        except ValueError:
            target = bits[0]
            collection = self.query.content or ''

        if collection == 'translated':
            collection = 'en'
        elif collection == 'raw':
            collection = ''

        return target, collection

    def collection_for_target(self, target):
        search, index = self.parse_target(target)

        return self.collection_for_target_pair(search, index)

    def collection_for_target_pair(self, search, index):
        try:
            collection, index = self.settings.base_collection(search, index)
        except KeyError:
            collection, index = self.settings.external_collection(search, index)

        return collection, index

    def endpoint_for_target(self, target):
        """Выбирает url для заданного поиска
        """
        search, index = self.parse_target(target)

        collection, index = self.collection_for_target(target)

        try:
            endpoint = collection['endpoint']
        except KeyError:
            if search == 'meta':
                revisions_by_service = {}

                revisions = (
                    self.revisions.find(source, index)
                    for source, index in self.settings.sources_for_meta(index)
                )
                revisions = sorted(revisions, key=lambda x: x['service'])

                for service, service_revisions in groupby(revisions, lambda x: x['service']):
                    revisions_by_service[service] = list(service_revisions)

                # берем первый сервис для получения инстанса
                service = next(revisions_by_service.keys())

                endpoint = deepcopy(settings.ISEARCH['api']['saas'][service]['search'])

                endpoint.setdefault('query', {})
                endpoint['query'].update(service=','.join(revisions_by_service))

                for service, revisions in revisions_by_service.items():
                    endpoint['query']['%s_kps' % service] = ','.join(str(get_kps(r['id'])) for r in revisions)
            else:
                revision = self.revisions.find(search, index)

                return self.endpoint_for_revision(revision)

        if 'request_builder' in collection:
            endpoint['request_builder'] = collection['request_builder']

        return endpoint

    def endpoint_for_revision(self, revision):
        """Формирует endpoint для переданной ревизии
        """
        collection, index = self.collection_for_target_pair(revision['search'],
                                                            revision['index'])
        service = revision['service']
        query = {'kps': get_kps(revision['id'])}

        endpoint = deepcopy(settings.ISEARCH['api']['saas'][service]['search'])

        endpoint.setdefault('query', {})
        endpoint['query'].update(query)

        if 'request_builder' in collection:
            endpoint['request_builder'] = collection['request_builder']

        return endpoint


class FormulaContext(BaseContext):
    '''
    Формулы
    '''
    def __init__(self, formulas=None, storage=None):
        super().__init__(storage)

        self.formulas = formulas or {}

    def __str__(self):
        return '<Formulas: {}>'.format(', '.join(f'{k} : {v}'
                                                        for k, v in self.formulas.items()))

    @classmethod
    def from_storage(cls, repository):
        storage = repository['formula']

        formulas = {(f['search'], f['index'], f['service'], f['name']): f
                                                        for f in storage.get()}

        return cls(formulas, storage=storage)

    @classmethod
    def from_raw(cls, data, **kwargs):
        formulas = {(f['search'], f['index'], f['service'], f['name']): f
                                                        for f in data}
        return cls(formulas, **kwargs)

    def get_formula(self, search, index, service, name):
        try:
            return self.formulas[(search, index, service, name)]
        except KeyError:
            return

    def current(self, search, index, service, name):
        for formula_name in (name, 'default'):
            for service_name in (service, ''):
                formula = self.get_formula(search, index, service_name, formula_name)
                if formula and formula['compiled']:
                    return (formula['compiled'], formula['additional'])
        return (None, None)

    def to_raw(self):
        return list(self.formulas.values())


class ExternalWizardRuleContext(BaseContext):
    '''
    Правила внешнего колдунщика
    '''
    def __init__(self, external_wizard_rules=None, storage=None):
        super().__init__(storage)

        self.external_wizard_rules = external_wizard_rules or {}

    @classmethod
    def from_storage(cls, repository):
        storage = repository['external_wizard_rule']
        rules = {(r['search'], r['index'], r['name']): r
                                                        for r in storage.get()}
        return cls(rules, storage=storage)

    @classmethod
    def from_raw(cls, data, **kwargs):
        rules = {(r['search'], r['index'], r['name']): r
                                                        for r in data}
        return cls(rules, **kwargs)

    def get(self, search, index, name):
        for rule_name in (name, 'default'):
            rule = self.external_wizard_rules.get((search, index, rule_name))
            if rule is not None:
                return (rule['rule'], dict(parse_qsl(rule['params'])))

    def to_raw(self):
        return list(self.external_wizard_rules.values())
