import json
import time
from logging import getLogger

from django.conf import settings

from intranet.search.core.utils import http


log = getLogger(__name__)


class FerrymanClient:
    """ Клиент для сервиса Ferryman от SaaS.
    Документация: https://wiki.yandex-team.ru/jandekspoisk/SaaS/Ferryman/
    """
    BATCH_STATUS_QUEUE = 'queue'
    BATCH_STATUS_PROCESSING = 'processing'
    BATCH_STATUS_FINAL = 'final'
    BATCH_STATUS_ERROR = 'error'

    def __init__(self, service='intrasearch'):
        self.conf = settings.ISEARCH['api']['ferryman']
        self.service = service

    def get_url(self, url_name):
        return self.conf[url_name].url().format(service=self.service)

    def add_table(self, kps, table, delta):
        data = [{"Path": str(table),
                 "Namespace": str(kps),
                 "Timestamp": int(round(time.time() * 1000000)),
                 "Delta": delta}]
        url = self.get_url('tables')
        params = {'tables': json.dumps(data)}
        data = self._do_request(url, params)
        return data['batch']

    def get_batch_status(self, batch):
        url = self.get_url('batch_status')
        data = self._do_request(url, {'batch': batch})
        return data

    def _do_request(self, url, params):
        try:
            session = http.create_session(max_retries=3)
            response = http.call_with_retry(session.get, url, params=params)
        except http.HTTPError as e:
            log.exception('Ferryman request failed: %s', e.response.text)
            raise
        except Exception:
            log.exception('Ferryman request failed')
            raise
        log.info('Ferryman request ok: %s', response.content)
        return response.json()
