import wtforms
from urllib.parse import unquote


class MaskField(wtforms.Field):
    """ Поле, позволяющее передавать значения в фиде <mask>.<name>=x
    Например, facet.city=2&facet.department=1.
    На выходе будет словарь вида {'city': 2, 'department': 1}
    """
    def __init__(self, mask, label=None, validators=None, separator='.', **kwargs):
        super().__init__(label, validators, **kwargs)
        self.data = {}
        self.mask = mask
        self.separator = separator

    def process(self, formdata, data=None):
        self.data = {}
        mask = self.mask + self.separator
        for key in formdata:
            if key.startswith(mask):
                field_name = unquote(key[len(mask):])
                self.data[field_name] = self.process_value(formdata, key)

    def process_value(self, formdata, key):
        return unquote(formdata.get(key))


class MultiValueMaskField(MaskField):
    """ Аналогично базовому классу, только может принимать несколько значений
    с одним и тем же именем
    Например, facet.city=2&facet.city=5&facet.department=1.
    На выходе будет словарь вида {'city': [2, 5], 'department': [1, ]}
    """
    def process_value(self, formdata, key):
        return [unquote(v) for v in formdata.getlist(key)]


class SplitField(wtforms.StringField):
    def __init__(self, label='', validators=None, coerce=str, **kwargs):
        self.coerce = coerce
        super().__init__(label, validators, **kwargs)

    def process_formdata(self, valuelist):
        super().process_formdata(valuelist)
        data = []
        for value in valuelist:
            if value:
                data += value.split(',')
        self.data = data


class SearchQueryField(wtforms.StringField):
    """ Поле текста поискового запроса
    """
    def process_formdata(self, valuelist):
        super().process_formdata(valuelist)
        self.data = self.data.strip()


class URLField(wtforms.StringField):
    """Поле url с опциональной схемой"""

    def process_formdata(self, valuelist):
        super().process_formdata(valuelist)
        if not self.data.startswith('http'):
            self.data = 'https://' + self.data
