import json

from collections import OrderedDict

from django.core.management.base import BaseCommand
from schematics.types import NumberType, BooleanType, DateTimeType, URLType, DateType
from schematics.types.compound import ModelType, ListType

from intranet.search.core.snippets.base import get_snippets

DEFAULT_TYPE = 'string'
TYPE_MAP = {
    NumberType: 'number',
    DateTimeType: 'datetime',
    DateType: 'date',
    URLType: 'url',
    BooleanType: 'boolean'
}


def get_field_type(field):
    for type, value in TYPE_MAP.items():
        if isinstance(field, type):
            return value
    return DEFAULT_TYPE


def dump_snippet(snippet, save_pure=False):
    result = OrderedDict()
    for field_name, field in snippet.fields.items():
        need_save_pure = save_pure or field_name in getattr(snippet, '_fields_to_highlight', [])
        print(field_name, need_save_pure)

        if isinstance(field, ModelType):
            result[field_name] = dump_snippet(field.model_class, need_save_pure)
        elif isinstance(field, ListType):
            dmp = (dump_snippet(field.model_class, need_save_pure)
                   if hasattr(field, 'model_class')
                   else get_field_type(field))
            result[field_name] = [dmp]
        else:
            result[field_name] = get_field_type(field)
            if need_save_pure:
                result['%s_pure' % field_name] = result[field_name]
    return result


class Command(BaseCommand):
    help = 'Dump snippet JSON Schema'

    def add_arguments(self, parser):
        parser.add_argument('search', type=str)

    def handle(self, search, **options):
        snippets = get_snippets(search)
        for snippet in snippets:
            print(json.dumps(dump_snippet(snippet), indent=4))
