import os

from django.conf import settings
from django.core.management.base import BaseCommand
from yt.wrapper import TablePath

from intranet.search.core.models import Mark
from intranet.search.core.storages.job_storage import job_status
from intranet.search.core.yt_client import client as yt


def serialize_mark(mark, allowed_keys):
    mark['created_at'] = mark['created_at'].isoformat()
    mark['updated_at'] = mark['updated_at'].isoformat()
    new_mark = {}
    for key in allowed_keys:
        new_mark[key] = mark.get(key)
    return new_mark


class Command(BaseCommand):
    help = 'Export marks to yt'

    def add_arguments(self, parser):
        parser.add_argument('-t', '--table', action='store',  default='',
                            help='YT table path for export'),

    allowed_keys = ['id', 'request_id', 'url', 'score', 'created_at', 'updated_at', 'uid']

    schema = [
        {'name': 'id', 'type': 'uint64', 'sort_order': 'ascending'},
        {'name': 'request_id', 'type': 'string'},
        {'name': 'url', 'type': 'string'},
        {'name': 'score', 'type': 'string'},
        {'name': 'created_at', 'type': 'string'},
        {'name': 'updated_at', 'type': 'string'},
        {'name': 'uid', 'type': 'string'}
    ]

    @job_status
    def handle(self, **options):
        path = self.get_table_path(options.get('table'))
        if not yt.exists(path):
            yt.create('table', path, recursive=True, attributes={'schema': self.schema})

        table = TablePath(path, append=True)
        last_mark_id = self.get_last_imported_mark_id(path)
        yt.write_table(table, self.get_marks(last_mark_id))

    def get_table_path(self, path=None):
        return path or os.path.join(settings.ISEARCH['yt']['base_path'], 'output', settings.YENV_TYPE, 'marks_new')

    def get_last_imported_mark_id(self, path):
        last_row = yt.get(f"{path}/@row_count") - 1
        if last_row < 0:
            return 0

        table = TablePath(path, start_index=last_row, columns=['id'])
        data = list(yt.read_table(table))
        return data[0]['id']

    def get_marks(self, last_mark_id=0):
        marks = Mark.objects.filter(id__gt=last_mark_id).exclude(uid__exact='').order_by('id').values()
        return [serialize_mark(m, self.allowed_keys) for m in marks]
