from collections import defaultdict
from datetime import timedelta

from django.conf import settings
from django.core.management.base import BaseCommand
from django.utils import timezone

from intranet.search.core.storages import RevisionStorage, FerrymanTableStorage
from intranet.search.core.storages.job_storage import job_status
from intranet.search.core.swarm.storage import YTDocumentStorage
from intranet.search.core.yt_client import client as yt
from intranet.search.yt.jobs.merge_doc_tables import Job


def get_max_push_hour():
    now = timezone.now()
    max_hour = now - timedelta(hours=1)
    return max_hour.strftime('%Y-%m-%dT%H:00:00')


class Command(BaseCommand):
    help = 'Команда отправки таблиц пушей в ферримен'

    @job_status
    def handle(self, **options):
        rev_storage = RevisionStorage()
        table_storage = FerrymanTableStorage()
        max_hour = get_max_push_hour()

        yt_tables = {table_storage.get_absolute_path(t, table_storage.push_path)
                        for t in yt.list(table_storage.push_path)}
        db_tables = set(table_storage.get_all_paths())
        tables_to_send = yt_tables - db_tables

        # нужно смержить таблицы одинаковых ревизий
        to_merge = defaultdict(list)
        for table in tables_to_send:
            data = table_storage.parse_table_path(table)
            if data.get('suffix') and data['suffix'] > max_hour:
                continue
            destination = table_storage.get_yt_table_path(search=data['search'], index=data['index'],
                                                          revision=data['revision'], pushes=True)
            YTDocumentStorage.convert_to_static(table)
            to_merge[destination].append(table)

        self.run_merge(to_merge)

        for table in to_merge.keys():
            data = table_storage.parse_table_path(table)
            storage = YTDocumentStorage(rev_storage.get(data['revision']))
            storage.flush(table=table, delta=True)

    def run_merge(self, tables):
        merge_job = Job({'env': settings.YENV_TYPE})
        merge_job.run(tables)
