from logging import getLogger

from django.core.management.base import BaseCommand

from intranet.search.core.models import Organization, Indexation
from intranet.search.core.swarm.tasks import reindex
from time import sleep


logger = getLogger(__name__)


class Command(BaseCommand):
    help = 'Аккуратная массовая переиндексация всех организаций. Запускает не больше N параллельных ' \
           'индексаций (по умолчанию 20)'

    def add_arguments(self, parser):
        parser.add_argument('search', type=str)
        parser.add_argument('--size', '-s', action='store', type=int, default=10,
                            help='batch size')
        parser.add_argument('--concurrency', '-c', action='store', type=int, default=10,
                            help='max count of simultaneous new indexations')
        parser.add_argument('--comment', action='store', type=str, default='',
                            help='task comment')
        parser.add_argument('--max_iterations', '-m', action='store', type=int, default=1,
                            help='max iterations count')
        parser.add_argument('--indexers', '-i', action='store', type=str,
                            default='directory:,directory:departments,directory:groups',
                            help='comma separated list of searches and indexes')
        parser.add_argument('--countdown', '-d', action='store', type=int, default=30,
                            help='sleep countdown between iterations')
        parser.add_argument('--new-revision', '-x', action='store_true', default=False,
                            help='create new revision')
        parser.add_argument('--from-org-id', '-o', action='store', type=int,
                            help='org directory id for starting')

    def reindex(self, orgs, **kwargs):
        for search, index in self.indexers:
            logger.info('START REINDEX %s.%s for %s', search, index, orgs)
            reindex(search, orgs, index=index, **kwargs)

    def handle(self, **options):
        self.indexers = [s.split(':') for s in options['indexers'].split(',')]
        orgs = Organization.objects.order_by('directory_id').values_list('directory_id', flat=True)
        if options['from_org_id']:
            orgs = orgs.filter(directory_id__gte=options['from_org_id'])
        orgs = list(orgs)

        if not orgs:
            logger.info('CANNOT FIND ANY ORGANIZATIONS')
            return

        start = 0
        i = 0

        while start < len(orgs):
            if Indexation.objects.filter(status='new').count() < options['concurrency']:
                bulk = orgs[start:start + options['size']]
                self.reindex(bulk, comment=options['comment'], new_revision=options['new_revision'])
                start = start + options['size']
            else:
                logger.info("WAIT")

            i += 1
            if i >= options['max_iterations']:
                break

            sleep(options['countdown'])
