import json

from django.conf import settings
from django.core.management.base import BaseCommand
from intranet.search.core import storages, context
from intranet.search.core.utils.saas import get_saas_config, get_all_factors

settings_context = context.SettingsContext()


def collect_zones(factors, search):
    return [f'z_{search}_{zone}' for zone in factors.get('zone', [])]


def format_zone_factors(zones):
    ZL_TYPE = 'ZL'
    types = ['BM25F', 'CM', 'CZL', 'CZ', 'IF']
    sub_types = ['St', 'Lm', 'Sy']
    zone_factors = []
    for zone in zones:
        for t in types:
            for s in sub_types:
                zone_factors.append(f'_{t}_{s}_{zone}')
        zone_factors.append(f'_{ZL_TYPE}_{zone}')
    return zone_factors


def format_static_factors(factors, search):
    return [f'STAT_{search}_{static}' for static in factors.get('static', [])]


def format_user_factors(factors, search):
    return [f'USER_{search}_{user}' for user in factors.get('user', [])]


def collect_service_searches(service):
    """ Собирает все индексаторы, которые хранятся в сервисе
    """
    indexers = set()

    # источник #1 - ревизии
    revision_storage = storages.RevisionStorage()
    revisions = revision_storage.get_for_service(service)

    for revision in revisions:
        indexers.add((revision['search'], revision['index']))

    # источник #2 - конфиг дефолтных поисков
    for search, data in settings.ISEARCH['searches']['base'].items():
        for index, index_data in data['indexes'].items():
            if index_data.get('default_service') == service:
                indexers.add((search, index))
    return indexers


def collect_factors(service):
    zones = set()
    meta_factors = {'zone': set(), 'static': set()}
    self_factors = {'zone': set(), 'static': set(), 'user': set()}

    for search, index in collect_service_searches(service):
        search_factors = settings_context.factors_for_search(search)
        s_zones = collect_zones(search_factors, search)
        zones.update(s_zones)
        self_factors['user'].update(format_user_factors(search_factors, search))
        self_factors['zone'].update(format_zone_factors(s_zones))
        self_factors['static'].update(format_static_factors(search_factors, search))

        search_meta_factors = settings_context.meta_factors_for_search(search, index)
        if search_meta_factors:
            m_zones = collect_zones(search_meta_factors, 'meta')
            zones.update(m_zones)
            meta_factors['zone'].update(format_zone_factors(m_zones))
            meta_factors['static'].update(format_static_factors(search_meta_factors, 'meta'))

    zones = list(zones)
    zones.sort()
    return zones, self_factors, meta_factors


class Command(BaseCommand):
    help = 'Generate relev.conf for saas service'
    relev_conf = None

    def add_arguments(self, parser):
        parser.add_argument('service', type=str)

    def handle(self, service, **options):
        self.service = service
        self.relev_conf = get_saas_config(service)

        zones, self_factors, meta_factors = collect_factors(service)
        max_id = self.update_relev_conf(meta_factors, self.get_max_id())
        max_id = self.update_relev_conf(self_factors, max_id)
        self.update_relev_conf({'dynamic': settings.ISEARCH['dynamic_factors']}, max_id)

        self.stdout.write(json.dumps(self.relev_conf, indent=4))

    def get_max_id(self):
        return max(get_all_factors(self.service).values())

    def update_relev_conf(self, factors, max_id):
        for factor_type, values in factors.items():
            for factor_name in values:
                conf_type = f'{factor_type}_factors'
                if factor_name in self.relev_conf[conf_type]:
                    continue

                max_id += 1
                if factor_type == 'static':
                    self.relev_conf[conf_type][factor_name] = {'default_value': 0, 'index': max_id}
                else:
                    self.relev_conf[conf_type][factor_name] = max_id
        return max_id
