from itertools import chain

import libra
from urllib.parse import unquote
from libra_parse_utils import Row, get_query_data, is_intrasearch, is_long_click


# считаем кликабельность только для первых десяти документов
CTR_MAX_POSTITION = 10


class Block:
    def __init__(self, zone, position, has_click=False, has_long_click=False, has_first_click=False):
        self.zone = zone
        self.position = position
        self.has_click = has_click
        self.has_long_click = has_long_click
        self.has_first_click = has_first_click


def gen_request_children(request):
    for block in request.GetMainBlocks():
        yield from block.GetChildren()


def get_zone_wizard_visual_conf(zone, wizard_name):
    """ Возвращает настройки визуального отображения в зависимости от зоны и типа колдунщика
    Подробности в https://st.yandex-team.ru/ISEARCH-5786

    Учитываем все документы и клики по ним как есть:
        - органическая выдача (zone=search_results)
        - колдунщик людей (wizard_name=people-wizard) - потому что показывается максимум 2 человека,
            их сниппеты по размеру соотвествуют сниппетам документов в органике (== полноразмерные)
        - колдунщик сервисов (wizard_name=services) - потому что показывается максимум 1 сервис
            в полноразмерном сниппете
        - колдунщик этушки (wizard_name=at-serp) - потому что показывается максимум 1 пост
            в полноразмерном сниппете

    Учитываем только первый документ и клики только по нему:
        - колдунщик рассылок (wizard_name=mldescription)
        - колдунщик клубов (wizard_name=clubs)
        Потому что у них показвается один первый сниппет, а остальное скрыто под кат.

    Учитываем только первые 2 документа и клики только по ним:
        - колдунщик переговорок (wizard_name=invite)
        - колдунщик оборудования (wizard_name=equipment)
        Потому что у них показвается максимум два сниппета, а остальное скрыто под кат.

    Учитываем первые три документа как один но считаем клики по ним всем:
        - колдунщик трекера (wizard_name=st-serp) и колдунщик stackoverflow (wizard_name=stackoverflow-serp)
        Потому что у них в одном полноразмерном сниппете отображается сразу три тикета/ответа:
        то есть в логе они выглядят как три полноценных документа, но на выдаче занимают место как
        один снипет.
    """

    conf = {'max_counted': None, 'count_all_clicks': False}

    if zone != 'search_results' and wizard_name in ('clubs', 'mldescription', 'st-serp'):
        conf['max_counted'] = 1
        if wizard_name == 'st-serp':
            conf['count_all_clicks'] = True
    if zone != 'search_results' and wizard_name in ('invite', 'equipment'):
        conf['max_counted'] = 2
    return conf


def get_request_blocks(request, data):
    blocks = []
    clicks = {}

    for number, click in enumerate(request.GetClicks()):
        clicks[unquote(click.Url)] = {'number': number, 'dwell_time': click.DwellTime}

    conf = get_zone_wizard_visual_conf(data['zone'], data['wizard_name'])

    for i, ch in enumerate(gen_request_children(request)):
        click = clicks.get(ch.Url)
        if conf['max_counted'] is not None and i >= conf['max_counted']:
            if conf['count_all_clicks'] and blocks:
                block = blocks[-1]
            else:
                break
        else:
            block = Block(zone=data['zone'], position=ch.Position)

        if click:
            block.has_click = True
            if is_long_click(click):
                block.has_long_click = True
            if click['number'] == 0:
                block.has_first_click = True
        blocks.append(block)

    return blocks, clicks


def regroup_requests(session):
    """ Перегруппировываем запросы по параметру intrasearch-reqid

    Это всё жесть и по-хорошему нужно просто класть в user_sessions свои логи с фронта:
    ровно то, что мы показали пользователю.
    """
    requests = {}
    for request in session:
        if not is_intrasearch(request) or request.PageNumber > 0:
            continue
        data = get_query_data(request)
        if not data or not data['scope'] or data['scope'] == 'suggest':
            continue

        # группируем запросы не по нашему reqid (при upstream errors на фронте у нас
        # могут быть одинаковые запросы с разными id), а по вертикали - текусту запроса и времени:
        # при тех же upstream error у нас в логах появляются дублирующиеся записи
        key = '-'.join([data['scope'], request.Query, str(request.Timestamp)])
        if key not in requests:
            requests[key] = {
                'reqid': data['reqid'],
                'scope': data['scope'],
                'request': request,
                'by_zones': {},
            }

        requests[key]['reqid'] = data['reqid']
        blocks, clicks = get_request_blocks(request, data)
        # из результатов берем те, по которым были клики, или какие-то (если кликов не было вообще).
        # это опять же подпорка под upstream error: те, где были клики,
        # точно показались пользователю, поэтому их и берем
        if clicks or not data['zone'] in requests[key]['by_zones']:
            requests[key]['by_zones'][data['zone']] = blocks

        if data['zone'] == 'search_results':
            requests[key]['request'] = request

    for key, req_data in requests.items():
        def visual_position_key(item):
            zone = item.zone
            position = item.position
            if zone == 'wizard':
                return position
            elif zone == 'search_results' and position < 3:
                return 10 + position
            elif zone == 'wizard_5':
                return 20 + position
            else:
                return 100 + position

        # собираем результаты всех зон в один отсортированный по визуальной позиции на выдаче список
        results = chain(*req_data['by_zones'].values())
        req_data['results'] = sorted(results, key=visual_position_key)

    return requests


def get_requests(blockstat_path, utils_path=''):
    def libra_callable(key, recs):
        top_last_position = 3
        session = libra.ParseSessionWithFat(recs, blockstat_path)

        for reqid, req_data in regroup_requests(session).items():
            request = req_data['request']

            results_count = 0
            results_organic_count = 0
            results_wizard_count = 0
            results_wizard5_count = 0

            organic_click_count = 0
            organic_long_click_count = 0
            organic_top3_click_count = 0
            organic_top3_long_click_count = 0
            first_organic_click_position = -1
            first_organic_long_click_position = -1
            click_on_organic_position = [0] * CTR_MAX_POSTITION

            wizard_click_count = 0
            wizard_long_click_count = 0

            wizard5_click_count = 0
            wizard5_long_click_count = 0

            click_count = 0
            long_click_count = 0
            top3_click_count = 0
            top3_long_click_count = 0

            for global_position, result in enumerate(req_data['results']):
                results_count += 1

                if result.has_click:
                    click_count += 1
                    if result.has_long_click:
                        long_click_count += 1
                    if global_position < top_last_position:
                        top3_click_count += 1
                        if result.has_long_click:
                            top3_long_click_count += 1
                if result.zone == 'search_results':
                    results_organic_count += 1
                    if result.has_click:
                        organic_click_count += 1
                        if result.position < CTR_MAX_POSTITION:
                            click_on_organic_position[result.position] += 1
                        if result.has_first_click:
                            first_organic_click_position = result.position
                        if result.has_long_click:
                            organic_long_click_count += 1
                            if result.has_first_click:
                                first_organic_long_click_position = result.position
                        if result.position < top_last_position:
                            organic_top3_click_count += 1
                            if result.has_long_click:
                                organic_top3_long_click_count += 1
                elif result.zone == 'wizard':
                    results_wizard_count += 1
                    if result.has_click:
                        wizard_click_count += 1
                        if result.has_long_click:
                            wizard_long_click_count += 1
                elif result.zone == 'wizard_5':
                    results_wizard5_count += 1
                    if result.has_click:
                        wizard5_click_count += 1
                        if result.has_long_click:
                            wizard5_long_click_count += 1

            ctr = {
                f'click_count_on_{i}': v for i, v in enumerate(click_on_organic_position)
            }
            yield Row(
                uid=key,
                reqid=request.ReqID,
                intrasearch_reqid=req_data['reqid'],
                scope=req_data['scope'],
                query=request.Query,
                req_ts=int(request.Timestamp or 0),
                results_count=int(results_count),
                results_organic_count=int(results_organic_count),
                results_wizard_count=int(results_wizard_count),
                results_wizard5_count=int(results_wizard5_count),
                organic_click_count=int(organic_click_count),
                organic_long_click_count=int(organic_long_click_count),
                organic_top3_click_count=int(organic_top3_click_count),
                organic_top3_long_click_count=int(organic_top3_long_click_count),
                first_organic_click_position=int(first_organic_click_position),
                first_organic_long_click_position=int(first_organic_long_click_position),
                wizard_click_count=int(wizard_click_count),
                wizard_long_click_count=int(wizard_long_click_count),
                wizard5_click_count=int(wizard5_click_count),
                wizard5_long_click_count=int(wizard5_long_click_count),
                click_count=int(click_count),
                long_click_count=int(long_click_count),
                top3_click_count=int(top3_click_count),
                top3_long_click_count=int(top3_long_click_count),
                **ctr
            )

    return libra_callable
