from copy import copy, deepcopy


def cls_name(ast_node):
    return ast_node.__class__.__name__


def get_kids(ast_node, with_empty=False):
    res = []
    for name in ('left', 'right'):
        if hasattr(ast_node, name):
            res.append(getattr(ast_node, name))
        else:
            if with_empty:
                res.append(None)
    return res


class ASTNode:
    priority = 100

    def __str__(self):
        return cls_name(self)

    def __eq__(self, value):
        def get_dict(item):
            return {i: getattr(item, i) for i in dir(item)
                    if not i.startswith('_') and not callable(getattr(item, i))}

        return get_dict(self) == get_dict(value)

    def __and__(self, other):
        return And(self, other)

    def __or__(self, other):
        return Or(self, other)

    def to_string(self):
        """ Преводит дерево в строку """
        def rec(node, parent):
            kids = get_kids(node, with_empty=True)
            empties = kids.count(None)

            if empties == 0:
                # значит это бинарная операция
                if node.priority > parent.priority:
                    tmplt = "(%s%s%s)"
                else:
                    tmplt = "%s%s%s"

                left, right = kids
                return tmplt % (rec(left, node), node._compile(), rec(right, node))
            elif empties == 1:
                # Унарная операция
                _, right = kids
                if node.priority > parent.priority:
                    tmplt = "(%s%s)"
                else:
                    tmplt = "%s%s"
                return tmplt % (node._compile(), rec(right, node))
            elif empties == 2:
                # значит мы в листе
                return node._compile()

        return rec(self, ASTNode()).strip()

    def format(self, **labels):
        """ Подставляет на место ноды с плейсхолдером дерево из labels
        и возвращает новое дерево.

        """
        new_node = copy(self)
        left, right = get_kids(self, with_empty=True)

        if left is not None:
            new_node.left = left.format(**labels)

        if right is not None:
            new_node.right = right.format(**labels)

        return new_node

    def filter_text_nodes(self):
        """ Возвращает итератор по тексту из текстовых нод """
        for node in get_kids(self):
            yield from node.filter_text_nodes()

    def clone(self):
        return deepcopy(self)


class _Word(ASTNode):
    """ Базовый класс для любых операндов """
    priority = 0

    def __init__(self, text):
        self.text = str(text)

    def __str__(self):
        return f'{cls_name(self)}({self.text})'

    def _compile(self):
        return self.text

    def __bool__(self):
        return bool(self.text)


class Text(_Word):
    def filter_text_nodes(self):
        yield self


class ParenthesizedText(Text):
    """ Текст в круглых скобках
    """
    def _compile(self):
        return '(%s)' % self.text


class OrderedText(Text):
    """ Текст в кавычках """
    def __str__(self):
        return f'{cls_name(self)}("{self.text}")'

    def _compile(self):
        return '"%s"' % self.text


class Keyword(_Word):
    """ Имя зоны или поискового атрибута """
    def map_to_internal(self, keyword):
        return keyword

    def _compile(self):
        return self.map_to_internal(self.text)


class WeightValue(_Word):
    pass


class _Operator(ASTNode):
    """ Базовый класс любого оператора """
    def _compile(self):
        return self.op


class _BinOp(_Operator):
    """ Базовый класс люого бинарного оператора """
    def __init__(self, left, right):
        if not isinstance(left, ASTNode):
            left = Keyword(left)
        self.left = left
        if not isinstance(right, ASTNode):
            right = Text(right)
        self.right = right

    def __str__(self):
        return "{}({}{}{})".format(
            cls_name(self), self.left, self._compile(), self.right)

    def _compile(self):
        return " %s " % self.op

    @classmethod
    def join(cls, nodes):
        """ Работает примерно как метод join у строк
        связывает набор нод бинарным оператором.

        Пример, если a, b, c это текстовые ноды, то:

        Or.join([a, b, c]).to_string() == "a | b | c"
        AndSoft.join([a, b, c]).to_string() == "a b c"
        Or.join([]) == None

        """
        nodes = iter(nodes)

        try:
            left = next(nodes)
        except StopIteration:
            return None

        for node in nodes:
            left = cls(left, node)

        return left


class _NonSpacedBinOp(_BinOp):
    """
    Базовый класс для бинарных операторов которые не должны быть
    заключены между пробелами
    """
    def _compile(self):
        return self.op


class AndNot(_BinOp):
    op = '~'
    priority = 6


class AndNotDoc(_BinOp):
    op = '~~'
    priority = 9


class And(_BinOp):
    op = '&'
    priority = 5


class AndDoc(_BinOp):
    op = '&&'
    priority = 8


class AndSoft(_NonSpacedBinOp):
    op = ' '
    priority = 7

    def __str__(self):
        return f"{cls_name(self)}({self.left} _ {self.right})"


class Or(_BinOp):
    op = '|'
    priority = 11


class Synonym(_BinOp):
    op = '^'
    priority = 5


class Constr(_BinOp):
    op = '<<'
    priority = 10


class Refine(_BinOp):
    op = '<-'
    priority = 12


class Rarr(_NonSpacedBinOp):
    op = '->'
    priority = 2


class AttrSearch(_NonSpacedBinOp):
    op = ':'
    priority = 2


class AttrLessSearch(AttrSearch):
    op = ':<'


class AttrLessEqualSearch(AttrSearch):
    op = ':<='


class AttrGreaterSearch(AttrSearch):
    op = ':>'


class AttrGreaterEqualSearch(AttrSearch):
    op = ':>='


class ZoneSearch(AttrSearch):
    """ Поиск текста в зоне """


class Weight(_NonSpacedBinOp):
    op = '::'
    priority = 1


class _UnaryOp(_Operator):
    """ Базовый класс для унарных операторов """
    def __init__(self, right):
        self.right = right

    def __str__(self):
        return f"{cls_name(self)}({self.op} {self.right})"


class ExactWord(_UnaryOp):
    op = '!'
    priority = 3


class ExactLemma(_UnaryOp):
    op = '!!'
    priority = 3


class _Dist(_BinOp):
    """ Базовый класс для любых операторов с расстоянием """
    def __init__(self, dist, left, right):
        self.start, self.stop = dist
        _BinOp.__init__(self, left, right)

    def _compile(self):
        return f' {self.op}({self.start} {self.stop}) '


class AndDist(_Dist):
    op = '&/'
    priority = 4


class SimpleAndDist(AndDist):
    op = '/'


class AndDocDist(_Dist):
    op = '&&/'
    priority = 7


class AndNotDist(_Dist):
    op = '~/'
    priority = 5


class AndNotDocDist(_Dist):
    op = '~~/'
    priority = 8


class Placeholder(ASTNode):
    """ Служебная нода, заменяется на настоящее дерево методом format """
    def __init__(self, label):
        self.label = label

    def _compile(self):
        raise ValueError('You cant compile tree with Placeholder node.')

    def __str__(self):
        return f'{cls_name(self)}({self.label})'

    def format(self, **labels):
        item = labels[self.label]

        if not isinstance(item, ASTNode):
            raise TypeError(
                "Expected type of label '%s' is ASTNode, not %s" %
                (self.label, type(item)))

        return deepcopy(item)
