from intranet.search.core.query import _Operator


class _FactorOp(_Operator):
    """
    Базовый класс любого оператора для подсчета пользовательских факторов.
    Подробнее о пользовательских факторах: https://wiki.yandex-team.ru/jandekspoisk/saas/quality/factors/user/
    """
    op = None

    def _value_to_string(self, value):
        try:
            return value.to_string()
        except AttributeError:
            return str(value)

    def to_string(self):
        return f'{self.op}({self.get_op_values()})'


class _UnaryOp(_FactorOp):
    """
    Базовый класс для унарной операции
    """
    def __init__(self, value):
        self.value = value

    def get_op_values(self):
        return self._value_to_string(self.value)


class _MultiValueOp(_FactorOp):
    """
    Базовый класс для операций, принимающих много значений
    """
    max_values = None

    def __init__(self, *args):
        self.values = args

    def get_op_values(self):
        return ','.join(self._value_to_string(v) for v in self.values)


class No(_UnaryOp):
    """
    Отрицание значения
    """
    op = 'no'


class Lt(_MultiValueOp):
    op = 'lt'
    max_values = 2


class Gt(_MultiValueOp):
    op = 'gt'
    max_values = 2


class And(_MultiValueOp):
    op = 'and'


class Or(_MultiValueOp):
    op = 'or'


class Min(_MultiValueOp):
    op = 'min'


class Max(_MultiValueOp):
    op = 'max'


class Sum(_MultiValueOp):
    op = 'sum'


class InSet(_FactorOp):
    """
    Проверка на нахождение значения атрибута (фактора или группового атрибута) в множесте значений
    """
    op = 'inset'

    def __init__(self, value, in_set_values):
        self.value = value
        self.in_set_values = list(in_set_values)

    def get_op_values(self):
        """ В inset нужно передавать просто все значения по порядку через запятую:
        inset(val, setval1, setval2, ...)
        """
        return ','.join(self._value_to_string(v) for v in [self.value] + self.in_set_values)


class InSetAny(InSet):
    """
    То же, что и inset, только не нулевой, если у проверяемого группового атрибута несколько значений
    """
    op = 'insetany'


class Eq(_MultiValueOp):
    """
    Простой костыль для проверки равенства двух значений
    """
    op = 'inset'
    max_values = 2


class Ln(_UnaryOp):
    op = 'ln'


class Log10(_UnaryOp):
    op = 'log10'
