from schematics.models import Model
from schematics.types import StringType, IntType, DateType, BooleanType
from schematics.types.compound import ListType, ModelType

from intranet.search.core.snippets.base import BaseSnippet, register, DomainChangingURLType
from intranet.search.core.sources.utils import get_document_url


class EarlyDateType(DateType):
    """ Тип для дат, которые могут быть меньше 1900 года
    """
    def to_primitive(self, value, context=None):
        # На некоторых платформах strftime не поддерживает
        # даты ниже 1900 года (https://bugs.python.org/issue1777412),
        # поэтому используем обычный isoformat
        return value.isoformat()


class DirectoryAdminUrlMixin(Model):
    """ Миксин, добавляющий ссылку на админку директории в сниппет
    """
    admin_url = DomainChangingURLType()

    obj_type = None
    obj_id_field = 'id'

    def validate(self, *args, **kwargs):
        super().validate(*args, **kwargs)
        # Добавляем admin_url явно, если его не было изначально в сниппете,
        # потому что мы не можем просто взять и всё переиндексировать в b2b

        # Поддерживаем ISEARCH-7356 и меняем ссылку на админку.
        # TODO: возможно, надо убрать в будущем
        if not self._data.get('admin_url'):
            # В тестинге и на престейбле пока что старая ссылка
            try:
                self._data['admin_url'] = get_document_url('admin').format(
                id=self._data[self.obj_id_field], type=self.obj_type)
            except:
                self._data['admin_url'] = get_document_url('admin')


class UserNameSnippet(Model):
    first = StringType()
    last = StringType()
    middle = StringType()


class UserDepartmentSnippet(Model):
    id = IntType(required=True)
    url = StringType()
    label = StringType()
    dep_name = StringType()
    description = StringType()
    maillists = ListType(StringType())


class UserGroupSnippet(BaseSnippet):
    id = IntType(required=True)
    type = StringType()
    label = StringType()
    name = StringType()
    description = StringType()
    email = StringType()


class ContactSnippet(Model):
    type = StringType(required=True)
    value = StringType(required=True)
    label = StringType()
    _type_forms = ListType(StringType)


class PhoneSnippet(Model):
    type = StringType(required=True)
    value = StringType(required=True)
    description = StringType()
    _type_forms = ListType(StringType)
    _forms = ListType(StringType)


class UserSnippet(DirectoryAdminUrlMixin, BaseSnippet):
    obj_type = 'users'
    obj_id_field = 'uid'

    uid = IntType(required=True)
    cloud_uid = StringType()
    url = DomainChangingURLType(required=True)
    login = StringType(required=True)
    doc_type = StringType()
    work_email = StringType()
    name = ModelType(UserNameSnippet, required=True)
    department = ModelType(UserDepartmentSnippet)
    groups = ListType(ModelType(UserGroupSnippet))
    contacts = ListType(ModelType(ContactSnippet))
    phones = ListType(ModelType(PhoneSnippet))
    about = StringType()
    gender = StringType()
    birthday = EarlyDateType()
    position = StringType()
    is_dismissed = BooleanType()

    _fields_to_highlight = ('login', 'name', 'department', 'groups', 'contacts',
                            'phones', 'about', 'position')


register('directory', UserSnippet)


class RelatedUserSnippet(Model):
    uid = IntType(required=True)
    login = StringType(required=True)
    name = ModelType(UserNameSnippet)
    url = DomainChangingURLType()


class DepartmentSnippet(DirectoryAdminUrlMixin, BaseSnippet):
    obj_type = 'departments'

    id = IntType()
    label = StringType()
    email = StringType()
    name = StringType()
    description = StringType()
    members_count = IntType()
    doc_type = StringType()
    head = ModelType(RelatedUserSnippet)


register('directory', DepartmentSnippet)
register('meta', DepartmentSnippet)


class GroupSnippet(DirectoryAdminUrlMixin, UserGroupSnippet):
    obj_type = 'groups'

    doc_type = StringType()
    admins = ListType(ModelType(RelatedUserSnippet))


register('directory', GroupSnippet)
register('meta', GroupSnippet)
