from datetime import datetime
import pytz
import re
import logging

import ids
from lxml import etree
from django.conf import settings
from ids.services.at.repositories.utils import is_club_id

from intranet.search.core.utils import http, reraise_as_recoverable
from intranet.search.core.sources.utils import get_persons, get_person, parse_html
from intranet.search.core.tvm import tvm2_client

DEFAULT_QU = 5

log = logging.getLogger(__name__)
session = http.create_session()


class UIDs:
    def __init__(self, cache):
        self.cache = cache

    def __getitem__(self, uid):
        uid = int(uid)

        data = self.cache.get(uid)
        if data:
            return data

        # проверка на клуб
        if is_club_id(uid):
            data = self.add_club(uid)
        else:
            data = self.add_user(uid)

        if data == {}:
            raise KeyError('Unknown uid %s' % uid)

        return data

    def __setitem__(self, key, value):
        self.cache.set(int(key), value)

    def __contains__(self, key):
        return self.cache.get(int(key)) is not None

    def add_user(self, uid):
        uid = int(uid)
        data = get_user_info(uid)
        if data is None:
            raise KeyError("Can't add user: %s" % uid)

        self[uid] = data
        return data

    def add_club(self, club_id):
        club_id = int(club_id)
        data = get_club_info(club_id)

        if data is None:
            raise KeyError("Can't add club: %s" % club_id)

        self[club_id] = data
        return data


def get_staff_users(lookup=None):
    if lookup is None:
        lookup = {}
    return get_persons(lookup)


def get_user_info(user):
    """ Возвращает информацию о пользователе в нужно формате """
    with reraise_as_recoverable(ids.exceptions.BackendError):
        try:
            if isinstance(user, str):
                user = get_person({'login': user})
            elif isinstance(user, int):
                user = get_person({'uid': str(user)})
        except ids.exceptions.BackendError as exc:
            if 400 <= exc.args[0] < 500:
                return {}

            raise

    data = {
        'login': user['login'],
        'name': "{} {}".format(
            user['name']['first'].get('ru', ''),
            user['name']['last'].get('ru', '')
        ),
        'name_en': "{} {}".format(
            user['name']['first'].get('en', ''),
            user['name']['last'].get('en', '')
        ),
        'qu': DEFAULT_QU,
    }
    return data


def get_club_info(club_id):
    """ Возвращает информацию о клубе. Ничего не кеширует. """
    club_info = get_data(
        settings.ISEARCH['api']['at']['club_info'].url().format(feed_id=club_id),
        _ok_statuses=(200, 404, 403)
    )

    return club_info or {}


def id_is_club(id):
    # id клубов в этушке лежат в отличном от айдишников
    # пользователей диапазоне и начинаются с 4611686018427387904
    return int(id) >= 2 ** 62


def get_id(long_id):
    return int(re.match(r'.*\/([0-9]*)', long_id).groups()[0])


def get_post_number(long_id):
    """ Длинный id поста выглядит так: urn:ya.ru:post/1120000000043277/34
    Номер поста - число после последнего слэша
    """
    return int(long_id.rsplit('/', 1)[-1])


def parse_long_id(long_id):
    uid, post_number = long_id.split('/')[-2:]
    return int(uid), int(post_number)


def repair_link(link):
    # этушка отдает урлы от ярушки, тут это чинится
    return link.replace('.ya.ru', '.at.yandex-team.ru')


def get_data(url, *args, **kwargs):
    """ Функция получения данных по урлу. Делает ретраи, возвращает
    джейсоны. Игнорирует 404. В случае неудачи возвращает None.

    """
    if 'timeout' not in kwargs:
        kwargs['timeout'] = 10

    if 'headers' not in kwargs:
        kwargs['headers'] = settings.ISEARCH['api']['at']['endpoint'].headers()
    kwargs['headers'][settings.TVM2_SERVICE_HEADER] = tvm2_client.get_service_ticket('at')

    ok_statuses = kwargs.pop('_ok_statuses', (200, 404))

    try:
        response = http.call_with_retry(session.get, url,
                                        _ok_statuses=ok_statuses,
                                        verify=False,
                                        *args, **kwargs)
    except http.ERRORS:
        log.exception("Can't get url %s", url)
        return

    if response.status_code in ok_statuses and not response.ok:
        # то значит это нормальная ситуация (например пользователь не
        # написал ни одного поста - это 404)
        return

    return response.json()


def get_posts(url):
    """ Возвращает итератор по всем постам """
    while url:
        data = get_data(url)
        if data is None:
            return
        yield from data.get('entries', [])
        url = data['links'].get('next')


def get_comments(uid, post_id):
    if is_club_id(uid):
        url = settings.ISEARCH['api']['at']['club_comments'].url().format(
            feed_id=uid, post_no=post_id)
    else:
        url = settings.ISEARCH['api']['at']['person_comments'].url().format(
            uid=uid, post_no=post_id)

    comments = get_data(url)

    if comments is None:
        return []
    return comments['replies']


def user_clubs(user):
    url = settings.ISEARCH['api']['at']['person_clubs'].url().format(uid=user)
    clubs = get_data(url)
    if clubs:
        return clubs['clubs']
    else:
        return []


def replace_ya(text, uids):
    """
    Заменяет <ya user="narn" uid="1120000000000074"/> на "Роман Андриади narn@"
    или <ya club="tools" uid="6...."/> на "Новости внутренних сервисов tools@".
    Такое форматирование встречается в постах до 2016 года.

    Формат тега ya:
        <ya
            user="tmalikova" - логин пользователя, если ссылка на пользователя
            club="tools" - слаг клуба, если ссылка на клуб
            uid="1234..." - uid клуба или пользователя, если есть, то в пост пишется
                            ИО пользователя или название клуба
            title="Тая" - замешающий текст, который пишется в пост
        >
    """
    if not text:
        return ''

    try:
        html = parse_html(text)
    except (etree.ParserError, etree.XMLSyntaxError):
        return text

    for node in html.xpath('//ya'):
        replace_text = []
        if 'title' in node.attrib:
            replace_text.append(node.attrib['title'])
        elif 'uid' in node.attrib:
            info = uids[int(node.attrib['uid'])]
            replace_text.append(info['name'])

        for field in ('user', 'club'):
            if field in node.attrib:
                replace_text.append(node.attrib[field] + '@')

        if replace_text:
            node.text = ' '.join(replace_text)

    return html


def parse_date(datestr):
    """ Парсит строчку в дату и возвращает ее с московской локалью """
    utc = pytz.timezone('UTC')
    moscow = pytz.timezone('Europe/Moscow')
    utc_time = utc.localize(datetime.strptime(datestr, "%Y-%m-%dT%H:%M:%SZ"))
    return utc_time.astimezone(moscow)


def get_marks(object):
    """
    Возвращает словарь с ключами likes и dislikes для поста.

    Этушка возвращает в словаре лайков два поля:
        - count - суммарное кол-во лайков (лайк - это 1, дислайк - -1), то есть ровно то,
        что отображается в счетчике лайков в посте
        - total - суммарное кол-во оценок (как лайков, так и дизлайков).
    """
    if object.get('likes') and object['likes'][0]:
        total = int(object['likes'][0]['total'])
        count = int(object['likes'][0]['count'])
        likes = (total + count) / 2
        dislikes = (total - count) / 2
    else:
        likes = dislikes = total = 0
    return {'likes': likes, 'dislikes': dislikes, 'total': total}
